/* tgl library
 * (c) 1999, Michel Beaudouin-Lafon, mbl@daimi.au.dk
 */

#ifdef nti
#include <windows.h>
#endif

#include <stdio.h>

#include <GL/gl.h>
#include "tgl.h"
#include "math.h"

/* *** hack for the Mac, M_PI is not in math.h, 
   and FillOval and FillRect exist in the toolbox */
#ifndef M_PI
#define M_PI 3.141592654
#endif
#define FillOval tglFillOval
#define FillRect tglFillRect
#define Point tglPoint

/* *** hack for microsoft, Arc is in the std API */
#ifdef nti
#define Arc tglArc
#endif

/* *** hack for our PCs: glColor4d just does not seem to work!!! */
#define glColor4d(r,g,b,a) glColor4f((float)(r),(float)(g),(float)(b),(float)(a))

/* local trig functions 
   *** these should be in trig.h and use a precompiled table */
#define sine sin
#define cosine cos

/* degrees to radians conversion */
#define radians(a) ((a)*M_PI/180.0)

/*
 * ---- declarations
 */

/* data structures */
typedef struct {
	GLdouble x, y;
} point;

typedef struct {
	point* p1;
	point* p2;
	GLdouble dx, dy;
	GLdouble norm;
	GLdouble nx, ny;
} segment;

/* state structures */
typedef struct _State {
	GLboolean inBeginEnd;
	GLint nslices;
	GLdouble slice;	/* angle per slice in radians: computed */
	GLenum style;
} State;

typedef struct _FillState {
	GLdouble border;
	GLdouble inR, inG, inB, inA;
	GLdouble outR, outG, outB, outA;
	GLdouble thickness;
} FillState;

typedef struct _ShadowState {
	GLdouble r, g, b;
	GLdouble aIn, aOut;
	GLdouble in, out;
	GLdouble dx, dy, dz;
	GLboolean fill;
} ShadowState;

typedef struct _PolyState {
	GLenum mode;
	GLint npoint;
	/* previous, current and next point */
	point P1, P, P2;
	/* previous and next segment */
	segment S1, S2;
	/* internal and external points */
	point Pin, Pout;
} PolyState;

typedef struct _ArrowState {
	GLenum mode;
	GLdouble length, width;
} ArrowState;

typedef struct _SavedState {
	State state;
	FillState fillState;
	ShadowState shadowState;
	PolyState polyState;	/* *** should this be saved ? */
	ArrowState arrowState;
} SavedState;

/* local functions */
void Point (point* p, GLdouble x, GLdouble y);
void OffsetPoint (point* p, GLdouble dx, GLdouble dy);
GLdouble Distance2 (point* p1, point* p2);
void Segment (segment* s, point* p1, point* p2);

int SegmentIntersect (segment* s, point* p1, point* p2, point* i);

void Arc (GLdouble x, GLdouble y, GLdouble w, GLdouble h, GLdouble from, GLdouble to);

void FillOval (GLdouble rx, GLdouble ry);
void BorderOval (GLdouble rx, GLdouble ry);
void ShadowOval (GLdouble rx, GLdouble ry);

void FillRect (GLdouble w2, GLdouble h2);
void BorderRect (GLdouble w2, GLdouble h2);
void ShadowRect (GLdouble w2, GLdouble h2);

int Corner (point* P, point* N1, point* N2, GLdouble e, point* M, point* M1, point* M2);

void FillLine ();
void BorderLine ();
void ShadowLine ();

void EndFillLine ();
void EndBorderLine ();
void EndShadowLine ();

GLboolean FillArrow (segment* s, GLenum mode, point* pP1, point* pP2);
GLboolean BorderArrow (segment* s, GLenum mode, point* pP1, point* pP2);
GLboolean ShadowArrow (segment* s, GLenum mode, point* pP1, point* pP2);


/*
 * ---- static variables
 */

static State state = {
	GL_FALSE,	/* inBeginEnd */
	60,		/* nslices */
	0.0,		/* slice - computed */
	TGL_BORDER	/* mode */
};

static FillState fillState = {
	0.2,		/* border */
	1.0, 1.0, 1.0, 1.0, 	/* inR, inG, inB, inA */
	0.9, 0.9, 0.9, 1.0, 	/* outR, outG, outB, outA */
	0.3		/* thickness */
};

static ShadowState shadowState = {
	0.1, 0.1, 0.1,	/* r, g, b */
	0.9, 0.0,	/* aIn, aOut */
	0.15, 0.1,	/* in, out */
	0.075, -0.075, 0.05,	/* dx, dy, dz */
	GL_FALSE	/* fill */
};

static PolyState polyState = {
	0,		/* mode */
	0		/* npoint */
};

static ArrowState arrowState = {
	TGL_ARROW_NONE,	/* mode */
	1.0, 0.5	/* length, width */
};

#define MAX_SAVED 10
static SavedState savedState [MAX_SAVED];
static int topState = 0;

/* dispatch tables */
typedef void (*OvalProc) (GLdouble, GLdouble);
static OvalProc ovalProc[TGL_NUMSTYLES] = {
	FillOval,
	BorderOval,
	ShadowOval
};

typedef void (*RectProc) (GLdouble, GLdouble);
static RectProc rectProc[TGL_NUMSTYLES] = {
	FillRect,
	BorderRect,
	ShadowRect
};

typedef void (*LineProc) ();
static LineProc lineProc[TGL_NUMSTYLES] = {
	FillLine,
	BorderLine,
	ShadowLine
};

typedef void (*EndLineProc) ();
static EndLineProc endLineProc[TGL_NUMSTYLES] = {
	EndFillLine,
	EndBorderLine,
	EndShadowLine
};

/*
 * ---- Initialization
 */
void tglInit ()
{
	state.slice = 2 * M_PI / state.nslices;
}

/*
 * ---- state management functions
 */

void tglPushAttrib ()
{
	if (state.inBeginEnd || topState >= MAX_SAVED)
		return;
	
	savedState[topState].state = state;
	savedState[topState].fillState = fillState;
	savedState[topState].shadowState = shadowState;
	savedState[topState].polyState = polyState;
	savedState[topState].arrowState = arrowState;
	topState++;
}

void tglPopAttrib ()
{
	if (state.inBeginEnd || topState <= 0)
		return;
	
	--topState;
	state = savedState[topState].state;
	fillState = savedState[topState].fillState;
	shadowState = savedState[topState].shadowState;
	polyState = savedState[topState].polyState;
	arrowState = savedState[topState].arrowState;
}

/* state for all drawing functions */
void tglNumSlices (GLint n)
{
	state.nslices = n;
	state.slice = 2 * M_PI / n;
}

void tglDrawStyle (GLenum s)
{
	if (state.inBeginEnd)
		return;
	
	if (s >= 0 && s < TGL_NUMSTYLES)
		state.style = s;
}

/* state for Fill functions */
void tglBorder (GLdouble b)
{
	fillState.border = b;
}

void tglFillColor3d (GLdouble r, GLdouble g, GLdouble b)
{
	fillState.inR = r;
	fillState.inG = g;
	fillState.inB = b;
	fillState.inA = 1.0;
}

void tglFillColor4d (GLdouble r, GLdouble g, GLdouble b, GLdouble a)
{
	fillState.inR = r;
	fillState.inG = g;
	fillState.inB = b;
	fillState.inA = a;
}

void tglBorderColor3d (GLdouble r, GLdouble g, GLdouble b)
{
	fillState.outR = r;
	fillState.outG = g;
	fillState.outB = b;
	fillState.outA = 1.0;
}

void tglBorderColor4d (GLdouble r, GLdouble g, GLdouble b, GLdouble a)
{
	fillState.outR = r;
	fillState.outG = g;
	fillState.outB = b;
	fillState.outA = a;
}

void tglThickness (GLdouble t)
{
	fillState.thickness = t;
}

/* state for Shadow functions */
void tglShadowColor3d (GLdouble r, GLdouble g, GLdouble b)
{
	shadowState.r = r;
	shadowState.g = g;
	shadowState.b = b;
}

void tglShadowTransparency (GLdouble ain, GLdouble aout)
{
	shadowState.aIn = ain;
	shadowState.aOut = aout;
}

void tglShadowThickness (GLdouble in, GLdouble out)
{
	shadowState.in = in;
	shadowState.out = out;
}

void tglShadowOffset (GLdouble dx, GLdouble dy, GLdouble dz)
{
	shadowState.dx = dx;
	shadowState.dy = dy;
	shadowState.dz = dz;
}

void tglShadowFill (GLboolean f)
{
	shadowState.fill = f;
}

/* state for Arrows */
void tglArrow (GLenum mode)
{
	arrowState.mode = mode;
}

void tglArrowSize (GLdouble l, GLdouble w)
{
	arrowState.length = l;
	arrowState.width = w;
}

/*
 * --- drawing functions
 */
void tglOval (GLdouble rx, GLdouble ry)
{
	if (state.inBeginEnd)
		return;
	
	(ovalProc[state.style]) (rx, ry);
}

void tglRect (GLdouble w2, GLdouble h2)
{
  if (state.inBeginEnd){
    fprintf(stderr, "tglRect: in begin/end - returning\n");
    return;
  }
  (rectProc[state.style]) (w2, h2);
}

void tglBegin (GLenum mode)
{
	if (state.inBeginEnd)
		return;
	
	polyState.mode = mode;
	state.inBeginEnd = GL_TRUE;
}

void tglVertex2d (GLdouble x2, GLdouble y2)
{
	if (! state.inBeginEnd)
		return;
	
	if (polyState.npoint == 0) {
		/* record first point */
		Point (&polyState.P1, x2, y2);
		polyState.npoint = 1;
		return;
	}
	
	if (polyState.npoint == 1) {
		if (x2 == polyState.P1.x && y2 == polyState.P1.y)
			return;
		
		/* record second point */
		Point (&polyState.P, x2, y2);
		Segment (&polyState.S1, &polyState.P1, &polyState.P);
		polyState.npoint = 2;
		
		return;
	}
	
	if (x2 == polyState.P.x && y2 == polyState.P.y)
		return;
		
	/* record third point */
	Point (&polyState.P2, x2, y2);
	Segment (&polyState.S2, &polyState.P, &polyState.P2);
	polyState.npoint++;
	
	/* draw line segment */
	(lineProc[state.style]) ();
	
	/* prepare for next point */
	polyState.P1 = polyState.P;
	polyState.P = polyState.P2;
	polyState.S1 = polyState.S2;
}

void tglEnd ()
{
	if (! state.inBeginEnd)
		return;
	
	state.inBeginEnd = GL_FALSE;
	
	if (polyState.npoint >= 2)
		(endLineProc[state.style]) ();
	
	polyState.npoint = 0;
	polyState.mode = 0;
}

/*
 * -------- end of extern functions
 */

/* macros to set GL state */
#define FillColorIn()	glColor4d (fillState.inR, fillState.inG, fillState.inB, fillState.inA)
#define FillColorOut()	glColor4d (fillState.outR, fillState.outG, fillState.outB, fillState.outA)
#define ShadowColorIn()	glColor4d (shadowState.r, shadowState.g, shadowState.b, shadowState.aIn)
#define ShadowColorOut()	glColor4d (shadowState.r, shadowState.g, shadowState.b, shadowState.aOut)

/* points and segments */

#define Vertex(p) glVertex2d ((p).x, (p).y)

static void Point (point* p, GLdouble x, GLdouble y)
{
	p->x = x;
	p->y = y;
}

static void OffsetPoint (point* p, GLdouble dx, GLdouble dy)
{
	p->x += dx;
	p->y += dy;
}

static GLdouble Distance2 (point* p1, point* p2)
{
	GLdouble dx = p2->x - p1->x;
	GLdouble dy = p2->y - p1->y;
	return dx*dx + dy*dy;
}

static void Segment (segment* s, point* p1, point* p2)
{
	s->p1 = p1;
	s->p2 = p2;
	s->dx = p2->x - p1->x;
	s->dy = p2->y - p1->y;
	s->norm = sqrt (s->dx*s->dx + s->dy*s->dy);
	if (s->norm != 0.0) {
		s->nx = -s->dy / s->norm;
		s->ny =  s->dx / s->norm;
	} else
		s->nx = s->ny = 0;
}


static int SegmentIntersect (segment* s, point* p1, point* p2, point* i)
{
	GLdouble dx = p2->x - p1->x;
	GLdouble dy = p2->y - p1->y;
	GLdouble dotprod = dx*s->nx + dy*s->ny;
	GLdouble t;
	
	/* if the dot product is 0, the segments are parallel
	   or one of the segments has zero length */
	if (dotprod == 0.0)
		return 0;
	
	/* parameter of intersection point on the p1-p2 segment */
	t = ((s->p1->x - p1->x)*s->nx + (s->p1->y - p1->y)*s->ny) / dotprod;
	
	/* if t is outside [0, 1], the intersection is outside the segment p1-p2 */
	if (t < 0.0 || t > 1.0)
		return 0;
	
	/* intersection point */
	i->x = p1->x + t*dx;
	i->y = p1->y + t*dy;
	
	/* parameter of intersection point on s. Use larger dimension for better accuracy */
	if (fabs(s->dx) > fabs(s->dy))
		t = (i->x - s->p1->x) / s->dx;
	else
		t = (i->y - s->p1->y) / s->dy;
	
	/* if t is outside [0, 1], the intersection is outside the segment s */
	if (t < 0.0 || t > 1.0)
		return 0;
	
	return 1;
}

static void Arc (GLdouble x, GLdouble y, GLdouble w, GLdouble h, GLdouble from, GLdouble to)
{
	if (from < to) {
		while (from < to) {
			glVertex2d (x + w*cosine(from), y + h*sine(from));
			from += state.slice;
		}
	} else {
		while (from > to) {
			glVertex2d (x + w*cosine(from), y + h*sine(from));
			from -= state.slice;
		}
	}
}

/*
 * ---- drawing ovals
 */
static void FillOval (GLdouble rx, GLdouble ry)
{
	int i;
	GLdouble a = state.slice;
	
	/* draw the oval as a triangle fan centered at the center of the oval */
	glBegin (GL_TRIANGLE_FAN);
	FillColorIn ();
	glVertex2d (0.0, 0.0);
	glVertex2d (rx, 0.0);
	for (i = 1; i < state.nslices; i++) {
		glVertex2d (rx*cosine(a), ry*sine(a));
		a += state.slice;
	}
	glVertex2d (rx, 0.0);
	glEnd ();
}

static void BorderOval (GLdouble rx, GLdouble ry)
{
	int i;
	GLdouble a = state.slice;
	GLdouble rxin = rx - fillState.border;
	GLdouble ryin = ry - fillState.border;
	
	if (fillState.border == 0.0 || rxin < 0.0 || ryin < 0.0) {
		/* draw the oval as a triangle fan centered at the center of the oval */
		glBegin (GL_TRIANGLE_FAN);
		FillColorIn ();
		glVertex2d (0.0, 0.0);
		FillColorOut ();
		glVertex2d (rxin, 0.0);
		for (i = 1; i < state.nslices; i++) {
			glVertex2d (rxin*cosine(a), ryin*sine(a));
			a += state.slice;
		}
		glVertex2d (rxin, 0.0);
		glEnd ();
		return;
	}
	
	/* draw the oval as a triangle fan centered at the center of the oval */
	glBegin (GL_TRIANGLE_FAN);
	FillColorIn ();
	glVertex2d (0.0, 0.0);
	glVertex2d (rxin, 0.0);
	for (i = 1; i < state.nslices; i++) {
		glVertex2d (rxin*cosine(a), ryin*sine(a));
		a += state.slice;
	}
	glVertex2d (rxin, 0.0);
	glEnd ();
	
	/* draw the darker (or lighter) border with a quadstrip */
	glBegin (GL_QUAD_STRIP);
	FillColorIn ();
	glVertex2d (rxin, 0.0);
	FillColorOut ();
	glVertex2d (rx, 0.0);
	a = state.slice;
	for (i = 1; i < state.nslices; i++) {
		FillColorIn ();
		glVertex2d (rxin*cosine(a), ryin*sine(a));
		FillColorOut ();
		glVertex2d (rx*cosine(a), ry*sine(a));
		a += state.slice;
	}
	FillColorIn ();
	glVertex2d (rxin, 0.0);
	FillColorOut ();
	glVertex2d (rx, 0.0);
	glEnd ();
}

static void ShadowOval (GLdouble rx, GLdouble ry)
{
	int i;
	GLdouble a = state.slice;
	GLdouble rxin = rx - shadowState.in;
	GLdouble ryin = ry - shadowState.in;
	
	glMatrixMode (GL_MODELVIEW);
	glPushMatrix ();
	glTranslated (shadowState.dx, shadowState.dy, shadowState.dz);
	
	rx += shadowState.out;
	ry += shadowState.out;
	
	/* draw the shadow with a quadstrip */
	glBegin (GL_QUAD_STRIP);
	ShadowColorIn ();
	glVertex2d (rxin, 0.0);
	ShadowColorOut ();
	glVertex2d (rx, 0.0);
	a = state.slice;
	for (i = 1; i < state.nslices; i++) {
		ShadowColorIn ();
		glVertex2d (rxin*cosine(a), ryin*sine(a));
		ShadowColorOut ();
		glVertex2d (rx*cosine(a), ry*sine(a));
		a += state.slice;
	}
	ShadowColorIn ();
	glVertex2d (rxin, 0.0);
	ShadowColorOut ();
	glVertex2d (rx, 0.0);
	glEnd ();
	
	/* fill the shadow if needed */
	if (shadowState.fill) {
		glBegin (GL_TRIANGLE_FAN);
		ShadowColorIn ();
		glVertex2d (0.0, 0.0);
		glVertex2d (rxin, 0.0);
		for (i = 1; i < state.nslices; i++) {
			glVertex2d (rxin*cosine(a), ryin*sine(a));
			a += state.slice;
		}
		glVertex2d (rxin, 0.0);
		glEnd ();
	}
	
	glPopMatrix ();
}

/*
 * ---- drawing rectangle
 */

static void FillRect (GLdouble w2, GLdouble h2)
{
	glBegin (GL_QUADS);
	FillColorIn ();
	glVertex2d (-w2, -h2);
	glVertex2d ( w2, -h2);
	glVertex2d ( w2,  h2);
	glVertex2d (-w2,  h2);
	glEnd ();
}

static void BorderRect (GLdouble w2, GLdouble h2)
{
	GLdouble leftIn;
	GLdouble rightIn;
	GLdouble top, bottom;
	
	if (fillState.border == 0.0 || fillState.border > w2 || fillState.border > h2) {
		/* draw as a triangle fan centered at the origin */
		glBegin (GL_TRIANGLE_FAN);
		FillColorIn ();
		glVertex2d (0.0, 0.0);
		FillColorOut ();
		glVertex2d (-w2, -h2);
		glVertex2d ( w2, -h2);
		glVertex2d ( w2,  h2);
		glVertex2d (-w2,  h2);
		glVertex2d (-w2, -h2);
		glEnd ();
		return;
	}
	
	/* divide the rectangle in a 3x3 rectangular mesh and draw it as 3 strips */
	leftIn = -w2 + fillState.border;
	rightIn = w2 - fillState.border;
	
	/* top strip */
	top = h2;
	bottom = h2 - fillState.border;
	glBegin (GL_QUAD_STRIP);
	FillColorOut ();
	glVertex2d (-w2, bottom);
	glVertex2d (-w2, top);
	FillColorIn ();
	glVertex2d (leftIn, bottom);
	FillColorOut ();
	glVertex2d (leftIn, top);
	FillColorIn ();
	glVertex2d (rightIn, bottom);
	FillColorOut ();
	glVertex2d (rightIn, top);
	glVertex2d (w2, bottom);
	glVertex2d (w2, top);
	glEnd ();
	
	/* middle strip */
	top = bottom;
	bottom = -h2 + fillState.border;
	glBegin (GL_QUAD_STRIP);
	FillColorOut ();
	glVertex2d (-w2, bottom);
	glVertex2d (-w2, top);
	FillColorIn ();
	glVertex2d (leftIn, bottom);
	glVertex2d (leftIn, top);
	glVertex2d (rightIn, bottom);
	glVertex2d (rightIn, top);
	FillColorOut ();
	glVertex2d (w2, bottom);
	glVertex2d (w2, top);
	glEnd ();
	
	/* bottom strip */
	top = bottom;
	bottom = -h2;
	glBegin (GL_QUAD_STRIP);
	FillColorOut ();
	glVertex2d (-w2, bottom);
	glVertex2d (-w2, top);
	glVertex2d (leftIn, bottom);
	FillColorIn ();
	glVertex2d (leftIn, top);
	FillColorOut ();
	glVertex2d (rightIn, bottom);
	FillColorIn ();
	glVertex2d (rightIn, top);
	FillColorOut ();
	glVertex2d (w2, bottom);
	glVertex2d (w2, top);
	glEnd ();
}

static void CheckError(char *file, int line)
{
  GLenum err=GL_NO_ERROR;
  err=glGetError(); 
  switch(err){
  case (GL_NO_ERROR):
    return;
  case (GL_INVALID_ENUM):
    fprintf(stderr, "%s:%d: GL error: GL_INVALID_ENUM (error 0x%x)\n", file, line, err);
    break;
  case (GL_INVALID_VALUE):
    fprintf(stderr, "%s:%d: GL error: GL_INVALID_VALUE (error 0x%x)\n", file, line, err);
    break;
  case (GL_INVALID_OPERATION):
    fprintf(stderr, "%s:%d: GL error: GL_INVALID_OPERATION (error 0x%x)\n", file, line, err);
    break;
  case (GL_STACK_OVERFLOW):
    fprintf(stderr, "%s:%d: GL error: GL_STACK_OVERFLOW (error 0x%x)\n", file, line, err);
    break;
  case (GL_STACK_UNDERFLOW):
    fprintf(stderr, "%s:%d: GL error: GL_STACK_UNDERFLOW (error 0x%x)\n", file, line, err);
    break;
  case (GL_OUT_OF_MEMORY):
    fprintf(stderr, "%s:%d: GL error: GL_OUT_OF_MEMORY (error 0x%x)\n", file, line, err);
    break;
  default:
    fprintf(stderr, "%s:%d: Unknown GL error: 0x%x\n", file, line, err);
    break;
  }
}

static void ShadowRect (GLdouble w2, GLdouble h2)
{
	GLdouble in = shadowState.in;
	GLdouble out = shadowState.out;
	
	/* to get a realistic shadow we need rounded corners */
	
	/* we tesselate the result into 4 triangle fans, each
	   centered at the inner corner of the shadows.
	   Each triangle fan starts with a triangle that covers
	   half of one of the borders adjacent to the center, then 
	   the triangles from the rounded corner (generated by Wedge) 
	   and ends with a triangle that covers half of the other
	   adjacent border. This is easier understood with a picture... */
#if 0
#define CK CheckError( __FILE__, __LINE__)
#else
#define CK
#endif

	CK;
	glMatrixMode (GL_MODELVIEW);
	CK;
	glPushMatrix ();
	CK;
	glTranslated (shadowState.dx, shadowState.dy, shadowState.dz);
	CK;
	
	/* bottom-left corner */
	glBegin (GL_TRIANGLE_FAN);
	ShadowColorIn ();
	glVertex2d (-w2 +in, -h2 +in);
	glVertex2d ( w2 -in, -h2 +in);
	ShadowColorOut ();
	glVertex2d (-w2 +in, -h2 -out);
	Arc (-w2, -h2, out, out, radians (270), radians (180));
	glVertex2d (-w2 -out, -h2 +in);
	glVertex2d (-w2 -out,  h2 -in);
	glEnd ();
	CK;
		
	/* top-left corner */
	glBegin (GL_TRIANGLE_FAN);
	ShadowColorIn ();
	glVertex2d (-w2 +in,  h2 -in);
	glVertex2d (-w2 +in, -h2 +in);
	ShadowColorOut ();
	glVertex2d (-w2 -out, h2 -in);
	Arc (-w2,  h2, out, out, radians (180), radians (90));
	glVertex2d (-w2 +in,  h2 +out);
	glVertex2d ( w2 -in,  h2 +out);
	glEnd ();
	CK;
	
	/* top-right corner */
	glBegin (GL_TRIANGLE_FAN);
	ShadowColorIn ();
	glVertex2d ( w2 -in,  h2 -in);
	glVertex2d (-w2 +in,  h2 -in);
	ShadowColorOut ();
	glVertex2d ( w2 -in,  h2 +out);
	Arc (w2,  h2, out, out, radians (90), radians (0));
	glVertex2d ( w2 +out,  h2 -in);
	glVertex2d ( w2 +out, -h2 +in);
	glEnd ();
	CK;
	
	/* bottom-right corner */
	glBegin (GL_TRIANGLE_FAN);
	ShadowColorIn ();
	glVertex2d ( w2 -in, -h2 +in);
	glVertex2d ( w2 -in,  h2 -in);
	ShadowColorOut ();
	glVertex2d ( w2 +out, -h2 +in);
	Arc (w2, -h2, out, out, radians (360), radians (270));
	glVertex2d ( w2 -in, -h2 -out);
	glVertex2d (-w2 +in, -h2 -out);
	glEnd ();
	CK;
	
	/* fill the shadow if needed */
	if (shadowState.fill) {
	  w2 -= in;
	  h2 -= in;
	  glBegin (GL_QUADS);
	  ShadowColorIn ();
	  glVertex2d (-w2, -h2);
	  glVertex2d ( w2, -h2);
	  glVertex2d ( w2,  h2);
	  glVertex2d (-w2,  h2);
	  glEnd ();
	  CK;
	} else {
	}
	CK;
	glPopMatrix ();
	CK;

}

/*
 * ---- drawing lines
 */

static int Corner (point* P, point* N1, point* N2, GLdouble e, point* M, point* M1, point* M2)
{
	GLdouble vecprod = N1->x*N2->y - N1->y*N2->x;
	GLdouble dx, dy, nx1, ny1, nx2, ny2, norm, xm, ym, cosa, cosphi;
	int retval;

	if (vecprod > 0) {
		/* point P is to the left of the segments (side pointed to by the normals) */
		retval = 1;
		nx1 = -N1->x; ny1 = -N1->y;
		nx2 = -N2->x; ny2 = -N2->y;
	} else {
		/* point P is to the right of the segments */
		retval = -1;
		nx1 = N1->x; ny1 = N1->y;
		nx2 = N2->x; ny2 = N2->y;
	}

	dx = -(nx1 + nx2);
	dy = -(ny1 + ny2);
	norm = sqrt (dx*dx + dy*dy);
	dx /= norm;
	dy /= norm;
	cosphi = nx1*nx2 + ny1*ny2;
	cosa = sqrt ((cosphi+1)/2);
	if (cosa != 0) {
		dx *= e/cosa;
		dy *= e/cosa;
	} else {
		dx = - nx1*e;
		dy = - ny1*e;
		retval = 0;
	}

	*M = *M1 = *M2 = *P;
	OffsetPoint (M, dx, dy);
	nx1 *= e; ny1 *= e;
	nx2 *= e; ny2 *= e;
	OffsetPoint (M1, nx1, ny1);
	OffsetPoint (M2, nx2, ny2);
/*	
glBegin (GL_TRIANGLES);
glColor3d (1.0, 0.0, 0.0);
Vertex (*M);
glColor3d (0.0, 1.0, 0.0);
Vertex (*M1);
glColor3d (0.0, 0.0, 1.0);
Vertex (*M2);
glEnd ();
   
glBegin (GL_LINES);
glColor3d (0.0, 1.0, 0.0);
Vertex (*P);
glVertex2d (P->x +N1->x, P->y +N1->y);
glColor3d (0.0, 0.0, 1.0);
Vertex (*P);
glVertex2d (P->x +N2->x, P->y +N2->y);
glEnd;
*/
    	return retval;
}

/* debugging: uncomment this to see outlines
#undef GL_TRIANGLE_FAN
#define GL_TRIANGLE_FAN GL_LINE_STRIP
*/

static void FillLine ()
{
	GLdouble t;
	GLdouble vecprod;
	GLdouble dx, dy;
	point M, M1, M2;
	
	if (! ((polyState.mode==TGL_LINE_STRIP) || (polyState.mode==TGL_LINE_LOOP)) )
		return;
	
	t = fillState.thickness;
	
	if (polyState.npoint == 3) {
		if (arrowState.mode & TGL_ARROW_START) {
			point P1, P2;
			
			if (! FillArrow (&polyState.S1, TGL_ARROW_START, &P1, &P2))
				P1 = polyState.P;
			polyState.P1 = P1;
			Segment (&polyState.S1, &P1, &polyState.P);
		}
		
		dx = t*polyState.S1.nx;
		dy = t*polyState.S1.ny;
		polyState.Pin = polyState.P1;
		polyState.Pout = polyState.P1;
		OffsetPoint (&polyState.Pin, -dx, -dy);
		OffsetPoint (&polyState.Pout, dx, dy);
	}
	
	M1 = polyState.Pin;
	OffsetPoint (&M1, polyState.S1.dx, polyState.S1.dy);
	M2 = polyState.Pout;
	OffsetPoint (&M2, polyState.S1.dx, polyState.S1.dy);
	
	vecprod = polyState.S1.dx*polyState.S2.dy - polyState.S1.dy*polyState.S2.dx;
	M = polyState.P;
	dx = t*polyState.S2.nx;
	dy = t*polyState.S2.ny;
	if (vecprod < 0) {
		OffsetPoint (&M, dx, dy);
	 	glBegin (GL_TRIANGLE_FAN);
	 	FillColorIn ();
	 	Vertex (polyState.P);
	 	Vertex (M1);
	 	Vertex (polyState.Pin);
	 	Vertex (polyState.Pout);
	 	Vertex (M2);
	 	Vertex (M);
	 	glEnd ();
	 	
	 	polyState.Pout = M;
	 	polyState.Pin = polyState.P;
	 	OffsetPoint (&polyState.Pin, -dx, -dy);
	} else {
		OffsetPoint (&M, -dx, -dy);
	 	glBegin (GL_TRIANGLE_FAN);
	 	FillColorIn ();
	 	Vertex (polyState.P);
	 	Vertex (M2);
	 	Vertex (polyState.Pout);
	 	Vertex (polyState.Pin);
	 	Vertex (M1);
	 	Vertex (M);
	 	glEnd ();
	 	
	 	polyState.Pin = M;
	 	polyState.Pout = polyState.P;
	 	OffsetPoint (&polyState.Pout, dx, dy);
	}
}

static void BorderLine ()
{
	GLdouble t;
	GLdouble vecprod;
	GLdouble dx, dy;
	point M, M1, M2;
	
	if (! ((polyState.mode==TGL_LINE_STRIP) || (polyState.mode==TGL_LINE_LOOP)) )
		return;
	
	/* *** border width is not used here to simplify computation */
	t = fillState.thickness;
	
	if (polyState.npoint == 3) {
		if (arrowState.mode & TGL_ARROW_START) {
			point P1, P2;
			
			if (! BorderArrow (&polyState.S1, TGL_ARROW_START, &P1, &P2))
				P1 = polyState.P;
			polyState.P1 = P1;
			Segment (&polyState.S1, &P1, &polyState.P);
		}
		
		dx = t*polyState.S1.nx;
		dy = t*polyState.S1.ny;
		polyState.Pin = polyState.P1;
		polyState.Pout = polyState.P1;
		OffsetPoint (&polyState.Pin, -dx, -dy);
		OffsetPoint (&polyState.Pout, dx, dy);
	}
	
	M1 = polyState.Pin;
	OffsetPoint (&M1, polyState.S1.dx, polyState.S1.dy);
	M2 = polyState.Pout;
	OffsetPoint (&M2, polyState.S1.dx, polyState.S1.dy);
	
	vecprod = polyState.S1.dx*polyState.S2.dy - polyState.S1.dy*polyState.S2.dx;
	M = polyState.P;
	dx = t*polyState.S2.nx;
	dy = t*polyState.S2.ny;
	if (vecprod < 0) {
		OffsetPoint (&M, dx, dy);
	 	glBegin (GL_TRIANGLE_FAN);
	 	FillColorIn ();
	 	Vertex (polyState.P);
	 	FillColorOut ();
	 	Vertex (M1);
	 	Vertex (polyState.Pin);
	 	FillColorIn ();
	 	Vertex (polyState.P1);
	 	FillColorOut ();
	 	Vertex (polyState.Pout);
	 	Vertex (M2);
	 	Vertex (M);
	 	glEnd ();
	 	
	 	polyState.Pout = M;
	 	polyState.Pin = polyState.P;
	 	OffsetPoint (&polyState.Pin, -dx, -dy);
	} else {
		OffsetPoint (&M, -dx, -dy);
	 	glBegin (GL_TRIANGLE_FAN);
	 	FillColorIn ();
	 	Vertex (polyState.P);
	 	FillColorOut ();
	 	Vertex (M2);
	 	Vertex (polyState.Pout);
	 	FillColorIn ();
	 	Vertex (polyState.P1);
	 	FillColorOut ();
	 	Vertex (polyState.Pin);
	 	Vertex (M1);
	 	Vertex (M);
	 	glEnd ();
	 	
	 	polyState.Pin = M;
	 	polyState.Pout = polyState.P;
	 	OffsetPoint (&polyState.Pout, dx, dy);
	}
}

static void ShadowLine ()
{
	GLdouble t;
	GLdouble vecprod;
	GLdouble dx, dy;
	point M, M1, M2;
	
	if (! ((polyState.mode==TGL_LINE_STRIP) || (polyState.mode==TGL_LINE_LOOP)) )
		return;

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix ();
	glTranslated (shadowState.dx, shadowState.dy, shadowState.dz);
	
	/* *** shadow is not properly softened */
	t = fillState.thickness + shadowState.out;
	
	if (polyState.npoint == 3) {
		if (arrowState.mode & TGL_ARROW_START) {
			point P1, P2;
			
			if (! ShadowArrow (&polyState.S1, TGL_ARROW_START, &P1, &P2))
				P1 = polyState.P;
			polyState.P1 = P1;
			Segment (&polyState.S1, &P1, &polyState.P);
		}
		
		dx = t*polyState.S1.nx;
		dy = t*polyState.S1.ny;
		polyState.Pin = polyState.P1;
		polyState.Pout = polyState.P1;
		OffsetPoint (&polyState.Pin, -dx, -dy);
		OffsetPoint (&polyState.Pout, dx, dy);
	}
	
	M1 = polyState.Pin;
	OffsetPoint (&M1, polyState.S1.dx, polyState.S1.dy);
	M2 = polyState.Pout;
	OffsetPoint (&M2, polyState.S1.dx, polyState.S1.dy);
	
	vecprod = polyState.S1.dx*polyState.S2.dy - polyState.S1.dy*polyState.S2.dx;
	M = polyState.P;
	dx = t*polyState.S2.nx;
	dy = t*polyState.S2.ny;
	if (vecprod < 0) {
		OffsetPoint (&M, dx, dy);
	 	glBegin (GL_TRIANGLE_FAN);
	 	ShadowColorIn ();
	 	Vertex (polyState.P);
	 	ShadowColorOut ();
	 	Vertex (M1);
	 	Vertex (polyState.Pin);
	 	ShadowColorIn ();
	 	Vertex (polyState.P1);
	 	ShadowColorOut ();
	 	Vertex (polyState.Pout);
	 	Vertex (M2);
	 	Vertex (M);
	 	glEnd ();
	 	
	 	polyState.Pout = M;
	 	polyState.Pin = polyState.P;
	 	OffsetPoint (&polyState.Pin, -dx, -dy);
	} else {
		OffsetPoint (&M, -dx, -dy);
	 	glBegin (GL_TRIANGLE_FAN);
	 	ShadowColorIn ();
	 	Vertex (polyState.P);
	 	ShadowColorOut ();
	 	Vertex (M2);
	 	Vertex (polyState.Pout);
	 	ShadowColorIn ();
	 	Vertex (polyState.P1);
	 	ShadowColorOut ();
	 	Vertex (polyState.Pin);
	 	Vertex (M1);
	 	Vertex (M);
	 	glEnd ();
	 	
	 	polyState.Pin = M;
	 	polyState.Pout = polyState.P;
	 	OffsetPoint (&polyState.Pout, dx, dy);
	}
	
	glPopMatrix ();
}

static void EndFillLine ()
{
	GLdouble t = fillState.thickness;
	GLdouble dx, dy;
	point M1, M2;
	
	if (! ((polyState.mode==TGL_LINE_STRIP) || (polyState.mode==TGL_LINE_LOOP)) )
		return;

	dx = t*polyState.S1.nx;
	dy = t*polyState.S1.ny;
	if (polyState.npoint == 2) {
		if (arrowState.mode != TGL_ARROW_NONE) {
			point P1, P2;
			
			if (! FillArrow (&polyState.S1, arrowState.mode, &P1, &P2))
				return;
			polyState.P1 = P1;
			polyState.P = P2;
			Segment (&polyState.S1, &P1, &P2);
		}
		
		polyState.Pin = polyState.P1; OffsetPoint (&polyState.Pin, -dx, -dy);
		polyState.Pout = polyState.P1; OffsetPoint (&polyState.Pout, dx, dy);
	} else {
		if (arrowState.mode & TGL_ARROW_END) {
			point P1, P2;
			
			if (! FillArrow (&polyState.S1, TGL_ARROW_END, &P1, &P2))
				P2 = polyState.P;
			polyState.P = P2;
			Segment (&polyState.S1, &polyState.P1, &P2);
		}
	}
	
	M1 = M2 = polyState.P;
	OffsetPoint (&M1, -dx, -dy);
	OffsetPoint (&M2, dx, dy);
	
	glBegin (GL_TRIANGLE_FAN);
	FillColorIn ();
	Vertex (polyState.P);
	Vertex (M1);
	Vertex (polyState.Pin);
	Vertex (polyState.Pout);
	Vertex (M2);
	glEnd ();
}

static void EndBorderLine ()
{
	GLdouble t = fillState.thickness;
	GLdouble dx, dy;
	point M1, M2;
	
	if (! ((polyState.mode==TGL_LINE_STRIP) || (polyState.mode==TGL_LINE_LOOP)) )
		return;
	
	/* *** border width is not used here to simplify computation */
	dx = t*polyState.S1.nx;
	dy = t*polyState.S1.ny;
	if (polyState.npoint == 2) {
		if (arrowState.mode != TGL_ARROW_NONE) {
			point P1, P2;
			
			if (! BorderArrow (&polyState.S1, arrowState.mode, &P1, &P2))
				return;
			polyState.P1 = P1;
			polyState.P = P2;
			Segment (&polyState.S1, &P1, &P2);
		}
		
		polyState.Pin = polyState.P1; OffsetPoint (&polyState.Pin, -dx, -dy);
		polyState.Pout = polyState.P1; OffsetPoint (&polyState.Pout, dx, dy);
	} else {
		if (arrowState.mode & TGL_ARROW_END) {
			point P1, P2;
			
			if (! BorderArrow (&polyState.S1, TGL_ARROW_END, &P1, &P2))
				P2 = polyState.P;
			polyState.P = P2;
			Segment (&polyState.S1, &polyState.P1, &P2);
		}
	}
	
	M1 = M2 = polyState.P;
	OffsetPoint (&M1, -dx, -dy);
	OffsetPoint (&M2, dx, dy);
	
	glBegin (GL_TRIANGLE_FAN);
	FillColorIn ();
	Vertex (polyState.P);
	FillColorOut ();
	Vertex (M1);
	Vertex (polyState.Pin);
	FillColorIn ();
	Vertex (polyState.P1);
	FillColorOut ();
	Vertex (polyState.Pout);
	Vertex (M2);
	glEnd ();
}

static void EndShadowLine ()
{
	GLdouble t = fillState.thickness + shadowState.out;
	GLdouble dx, dy;
	point M1, M2;
	
	if (! ((polyState.mode==TGL_LINE_STRIP) || (polyState.mode==TGL_LINE_LOOP)) )
		return;
	
	glMatrixMode (GL_MODELVIEW);
	glPushMatrix ();
	glTranslated (shadowState.dx, shadowState.dy, shadowState.dz);
	
	/* *** shadow is not properly softened */
	dx = t*polyState.S1.nx;
	dy = t*polyState.S1.ny;
	if (polyState.npoint == 2) {
		if (arrowState.mode != TGL_ARROW_NONE) {
			point P1, P2;
			
			if (! ShadowArrow (&polyState.S1, arrowState.mode, &P1, &P2))
				return;
			polyState.P1 = P1;
			polyState.P = P2;
			Segment (&polyState.S1, &P1, &P2);
		}
		
		polyState.Pin = polyState.P1; OffsetPoint (&polyState.Pin, -dx, -dy);
		polyState.Pout = polyState.P1; OffsetPoint (&polyState.Pout, dx, dy);
	} else {
		if (arrowState.mode & TGL_ARROW_END) {
			point P1, P2;
			
			if (! ShadowArrow (&polyState.S1, TGL_ARROW_END, &P1, &P2))
				P2 = polyState.P;
			polyState.P = P2;
			Segment (&polyState.S1, &polyState.P1, &P2);
		}
	}
	
	M1 = M2 = polyState.P;
	OffsetPoint (&M1, -dx, -dy);
	OffsetPoint (&M2, dx, dy);
	
	glBegin (GL_TRIANGLE_FAN);
	ShadowColorIn ();
	Vertex (polyState.P);
	ShadowColorOut ();
	Vertex (M1);
	Vertex (polyState.Pin);
	ShadowColorIn ();
	Vertex (polyState.P1);
	ShadowColorOut ();
	Vertex (polyState.Pout);
	Vertex (M2);
	glEnd ();
	
	glPopMatrix ();
}

/*
 * ---- drawing arrows
 */

static GLboolean FillArrow (segment* s, GLenum mode, point* pP1, point* pP2)
{
	point P, M, M1, M2;
	GLdouble dx, dy;
	GLdouble dwx = arrowState.width*s->nx;
	GLdouble dwy = arrowState.width*s->ny;
	
	if (s->norm == 0.0)
		return GL_FALSE;
	
	if (mode == TGL_ARROW_BOTH) {
		point P1, P2;
		
		/* *** need a better way to do this */
		FillArrow (s, TGL_ARROW_START, pP1, &P2);
		FillArrow (s, TGL_ARROW_END, &P1, pP2);
		return GL_TRUE;
	}
	
	dx = arrowState.length * s->dx / s->norm;
	dy = arrowState.length * s->dy / s->norm;
	
	if (mode & TGL_ARROW_START) {
		P = M = *s->p1;
		OffsetPoint (&M, dx, dy);
		*pP1 = M;
	} else {
		*pP1 = *s->p1;
	}
	
	if (mode & TGL_ARROW_END) {
		P = M = *s->p2;
		OffsetPoint (&M, -dx, -dy);
		*pP2 = M;
	} else {
		*pP2 = *s->p2;
	}
	
	M1 = M2 = M;
	OffsetPoint (&M1, dwx, dwy);
	OffsetPoint (&M2, -dwx, -dwy);
	
	glBegin (GL_TRIANGLE_FAN);
	FillColorIn ();
	Vertex (P);
	Vertex (M1);
	Vertex (M2);
	glEnd ();
	
	/* check if the remaining segment is empty */
	/* *** for some reason this does not work as expected */
#if 0
	dx = pP2->x - pP1->x;
	dy = pP2->y - pP1->y;
	if (dx*s->dx < 0.0 || dy*s->dy < 0.0)
		return GL_FALSE;
#endif
	
	return GL_TRUE;
}

static GLboolean BorderArrow (segment* s, GLenum mode, point* pP1, point* pP2)
{
	point P, M, M1, M2;
	GLdouble dx, dy;
	GLdouble dwx = arrowState.width*s->nx;
	GLdouble dwy = arrowState.width*s->ny;
	
	if (s->norm == 0.0)
		return GL_FALSE;
	
	if (mode == TGL_ARROW_BOTH) {
		point P1, P2;
		
		/* *** need a better way to do this */
		BorderArrow (s, TGL_ARROW_START, pP1, &P2);
		BorderArrow (s, TGL_ARROW_END, &P1, pP2);
		return GL_TRUE;
	}
	
	dx = arrowState.length * s->dx / s->norm;
	dy = arrowState.length * s->dy / s->norm;
	
	/* *** border width is not used here to simplify computation */
	if (mode & TGL_ARROW_START) {
		P = M = *s->p1;
		OffsetPoint (&M, dx, dy);
		*pP1 = M;
	} else {
		*pP1 = *s->p1;
	}
	
	if (mode & TGL_ARROW_END) {
		P = M = *s->p2;
		OffsetPoint (&M, -dx, -dy);
		*pP2 = M;
	} else {
		*pP2 = *s->p2;
	}
	
	M1 = M2 = M;
	OffsetPoint (&M1, dwx, dwy);
	OffsetPoint (&M2, -dwx, -dwy);
	
	glBegin (GL_TRIANGLE_FAN);
	FillColorOut ();
	Vertex (P);
	Vertex (M1);
	FillColorIn ();
	Vertex (M);
	FillColorOut ();
	Vertex (M2);
	glEnd ();
	
	/* check if the remaining segment is empty */
	/* *** for some reason this does not work as expected */
#if 0
	dx = pP2->x - pP1->x;
	dy = pP2->y - pP1->y;
	if (dx*s->dx < 0.0 || dy*s->dy < 0.0)
		return GL_FALSE;
#endif
	
	return GL_TRUE;
}


static GLboolean ShadowArrow (segment* s, GLenum mode, point* pP1, point* pP2)
{
	point P, M, M1, M2;
	segment S1, S2;
	point N, N1, N2, P1, P2, Pin, M1in, M2in, M11, M12, M21, M22;
	GLdouble dx, dy;
	GLdouble dwx = arrowState.width*s->nx;
	GLdouble dwy = arrowState.width*s->ny;
	
	if (s->norm == 0.0)
		return GL_FALSE;
	
	if (mode == TGL_ARROW_BOTH) {
		/* *** need a better way to do this */
		ShadowArrow (s, TGL_ARROW_START, pP1, &P2);
		ShadowArrow (s, TGL_ARROW_END, &P1, pP2);
		return GL_TRUE;
	}
	
	N.x = s->dx / s->norm;
	N.y = s->dy / s->norm;
	dx = arrowState.length * N.x;
	dy = arrowState.length * N.y;
	
	/* *** border width is not used here to simplify computation */
	if (mode & TGL_ARROW_START) {
		P = M = *s->p1;
		OffsetPoint (&M, dx, dy);
		*pP1 = M;
	} else {
		*pP1 = *s->p1;
	}
	
	if (mode & TGL_ARROW_END) {
		P = M = *s->p2;
		OffsetPoint (&M, -dx, -dy);
		*pP2 = M;
	} else {
		*pP2 = *s->p2;
	}
	
	M1 = M2 = M;
	OffsetPoint (&M1, dwx, dwy);
	OffsetPoint (&M2, -dwx, -dwy);
	
	N.x = -N.x;
	N.y = -N.y;
	Segment (&S1, &M1, &P);
	Segment (&S2, &P, &M2);
	N1.x = S1.nx;
	N1.y = S1.ny;
	N2.x = S2.nx;
	N2.y = S2.ny;
	
	/* *** Corner should take outside and inside shadow widths... */
	Corner (&M1, &N, &N1, shadowState.out, &M1in, &M11, &M12);
	Corner (&P, &N1, &N2, shadowState.out, &Pin, &P1, &P2);
	Corner (&M2, &N2, &N, shadowState.out, &M2in, &M21, &M22);
	
	glBegin (GL_TRIANGLE_FAN);
	ShadowColorIn ();
	Vertex (M1in);
	Vertex (M2in);
	ShadowColorOut ();
	Vertex (M11);
	Vertex (M12);
	Vertex (P1);
	glEnd ();
	
	glBegin (GL_TRIANGLE_FAN);
	ShadowColorIn ();
	Vertex (Pin);
	Vertex (M1in);
	ShadowColorOut ();
	Vertex (P1);
	Vertex (P2);
	Vertex (M21);
	glEnd ();
	
	glBegin (GL_TRIANGLE_FAN);
	ShadowColorIn ();
	Vertex (M2in);
	Vertex (Pin);
	ShadowColorOut ();
	Vertex (M21);
	Vertex (M22);
	Vertex (M11);
	glEnd ();
	
	/* *** fill the shadow triangle since we can't use inside shadow width */
	glBegin (GL_TRIANGLE_FAN);
	ShadowColorIn ();
	Vertex (M1in);
	Vertex (Pin);
	Vertex (M2in);
	glEnd ();
	
	/* check if the remaining segment is empty */
	/* *** for some reason this does not work as expected */
#if 0
	dx = pP2->x - pP1->x;
	dy = pP2->y - pP1->y;
	if (dx*s->dx < 0.0 || dy*s->dy < 0.0)
		return GL_FALSE;
#endif
	
	return GL_TRUE;
}

