#ifdef nti
#include <windows.h>
#endif

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>

#include <stdio.h>

#include "tgl.h"

#define DISK 1
#define BOX 2
#define EDGE 3
#define PEDGE 4
#define POLY 5

#define MAXPTS 20

typedef struct _Object{
	GLfloat x, y, z;
	int type;
	/* for edges */
	struct _Object* from;
	struct _Object* to;
	/* for polygons */
	int npoints;
	int arrows;
	GLfloat X[MAXPTS];
	GLfloat Y[MAXPTS];
} Object;

#define MAXOBJ 200
static Object* Objects[MAXOBJ];

int Lighting = 0;
int ShowPoints = 0;
int Wireframe = 0;
int NSteps = 50;

Object* CurPoly = 0;
Object* Halo = 0;

extern GLfloat VIEWSIZE;

/* --------------------------------------------------------------------------------- */
/* --------------------------------------------------------------------------------- */

void
SetMaterial (float r, float g, float b)
{
	if (Lighting) {
		GLfloat mat_diffuse[4];
		GLfloat mat_specular[] = {1.0, 1.0, 1.0, 1.0};
		GLfloat mat_shininess[] = {50.0};
		
		mat_diffuse[0] = r;
		mat_diffuse[1] = g;
		mat_diffuse[2] = b;
		mat_diffuse[3] = 1.0;
		
		glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, mat_diffuse);
		glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
		glMaterialfv (GL_FRONT_AND_BACK, GL_SHININESS, mat_shininess);
	} else {
		glColor3f (r, g, b);
	    	tglFillColor3d (r, g, b);
	    	tglBorderColor3d (r/1.1, g/1.1, b/1.1);
	}
}

void
Module_Resize(int width, int height)
{
	glMatrixMode(GL_PROJECTION) ;
	glLoadIdentity() ;
	glOrtho(-VIEWSIZE, VIEWSIZE, -VIEWSIZE, VIEWSIZE, -VIEWSIZE, VIEWSIZE) ;
	glViewport(0,0,width,height) ;
}

void
menuEvents (int choice)
{
	switch (choice) {
	case 0:	exit (0); break;
	case 1:	Lighting = !Lighting; break;
	case 2:	ShowPoints = !ShowPoints; break;
	case 10:	Wireframe = !Wireframe; break;
	}
	
	if (choice > 1000) {
		VIEWSIZE = choice - 1000;
		glMatrixMode(GL_PROJECTION) ;
		glLoadIdentity() ;
		glOrtho(-VIEWSIZE, VIEWSIZE, -VIEWSIZE, VIEWSIZE, -VIEWSIZE, VIEWSIZE) ;
	} else if (choice > 100) {
		NSteps = choice - 100;
		tglNumSlices (NSteps);
	}
	
	glutPostRedisplay ();
}

Object*
NewObj (GLfloat x, GLfloat y, int type)
{
	Object* obj;
	int i;

	for (i = 0; i < MAXOBJ; i++)
		if (! Objects[i])
			break;

	if (i == MAXOBJ)
		return 0;
	
	printf ("Creating obj #%d of type %d at %g %g\n", i, type, x, y);
	
	obj = (Object*) malloc (sizeof (Object));
	obj->x = x;
	obj->y = y;
	obj->z = 0.0;
	obj->type = type;
	obj->from = obj->to = 0;
	obj->arrows = TGL_ARROW_NONE;
	obj->npoints = 0;
	
	Objects[i] = obj;
	return obj;
}

Object*
NewEdge (Object* from, Object* to)
{
	Object* edge;
	
	if (! from || ! to)
		return 0;
	edge = NewObj (0, 0, EDGE);
	edge->from = from;
	edge->to = to;
	return edge;
}

Object*
NewPEdge (Object* from, Object* to, GLfloat x, GLfloat y)
{
	Object* edge = NewEdge (from, to);
	
	if (! edge)
		return 0;
	edge->type = PEDGE;
	edge->x = x;
	edge->y = y;
	return edge;
}

Object*
NewPoly (GLfloat x, GLfloat y)
{
	Object* poly = NewObj (0, 0, POLY);
	
	CurPoly = poly;
	poly->npoints = 1;
	poly->X[0] = x;
	poly->Y[0] = y;
	return poly;
}

void
AddPoint (GLfloat x, GLfloat y)
{
	if (! CurPoly)
		return;
	
	if (CurPoly->npoints >= MAXPTS)
		return;
	
	CurPoly->X[CurPoly->npoints] = x - CurPoly->x;
	CurPoly->Y[CurPoly->npoints] = y - CurPoly->y;
	CurPoly->npoints++;
}

void
EnableBlending ()
{
	glEnable (GL_BLEND);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask (GL_FALSE);
}

void
DisableBlending ()
{
	glDisable (GL_BLEND);
	glDepthMask (GL_TRUE);
}

void RenderShadow (Object* obj)
{
	EnableBlending ();
	glMatrixMode (GL_MODELVIEW) ;
	glPushMatrix() ;
	if (obj->type != PEDGE)
		glTranslated(obj->x, obj->y, 0.0) ;

	tglDrawStyle (TGL_SHADOW);
	switch (obj->type) {
	    case DISK: {
	    	tglOval (1.0, 1.0);
		break;
	    }
	    case BOX: {
	    	tglRect (1.0, 0.8);
		break;
	    }
	    case EDGE: {
	    	tglBegin (TGL_LINE_STRIP);
	    	tglVertex2d (obj->from->x, obj->from->y);
	    	tglVertex2d (obj->to->x, obj->to->y);
	    	tglEnd ();
	    	break;
	    }
	    case PEDGE: {
	    	tglBegin (TGL_LINE_STRIP);
	    	tglVertex2d (obj->from->x, obj->from->y);
	    	tglVertex2d (obj->x, obj->y);
	    	tglVertex2d (obj->to->x, obj->to->y);
	    	tglEnd ();
	    	break;
	    }
	    case POLY: {
	    	int i;
	    	
	    	tglArrow (obj->arrows);
	    	tglBegin (TGL_LINE_STRIP);
	    	for (i = 0; i < obj->npoints; i++)
	    		tglVertex2d (obj->X[i], obj->Y[i]);
	    	tglEnd ();
	    	tglArrow (TGL_ARROW_NONE);
	    	break;
	    }
	}
	tglDrawStyle (TGL_BORDER);
	
	glPopMatrix ();
	DisableBlending ();
}

void
RenderObject (GLenum mode, Object* obj)
{
	glMatrixMode (GL_MODELVIEW) ;
	glPushMatrix() ;
	if (obj->type == PEDGE)
		glTranslated(0.0, 0.0, obj->z) ;
	else
		glTranslated(obj->x, obj->y, obj->z) ;
	
	switch (obj->type) {
	    case DISK: {
	    	tglOval (1.0, 1.0);
		break;
	    }
	    case BOX: {
	    	tglRect (1.0, 0.8);
	    	break;
	    }
	    case EDGE: {
	    	tglBegin (TGL_LINE_STRIP);
	    	tglVertex2d (obj->from->x, obj->from->y);
	    	tglVertex2d (obj->to->x, obj->to->y);
	    	tglEnd ();
	    	break;
	    }
	    case PEDGE: {
	    	tglBegin (TGL_LINE_STRIP);
	    	tglVertex2d (obj->from->x, obj->from->y);
	    	tglVertex2d (obj->x, obj->y);
	    	tglVertex2d (obj->to->x, obj->to->y);
	    	tglEnd ();
	    	break;
	    }
	    case POLY: {
	    	int i;
	    	
	    	tglArrow (obj->arrows);
	    	tglBegin (TGL_LINE_STRIP);
	    	for (i = 0; i < obj->npoints; i++)
	    		tglVertex2d (obj->X[i], obj->Y[i]);
	    	tglEnd ();
	    	tglArrow (TGL_ARROW_NONE);
	    	break;
	    }
	}
	
	glPopMatrix() ;
}

void
Module_Draw(GLenum mode, int obj)
{
	int i ;
	
	if (Lighting && mode != GL_SELECT)
		glEnable (GL_LIGHTING);
	else
		glDisable (GL_LIGHTING);
	
	if (ShowPoints) {
		for (i=0; i<MAXOBJ; i++) {
			if (! Objects[i])
				continue;
			if (i == obj)
				SetMaterial (1.0, 0.4, 0.4);
			else
				SetMaterial (.8, .8, 1.0) ;
			
			if (mode == GL_SELECT)
				glPushName (i);
			glMatrixMode (GL_MODELVIEW) ;
			glPushMatrix() ;
			glTranslated(Objects[i]->x, Objects[i]->y, Objects[i]->z) ; 
			glutSolidCube(0.2) ;
			glPopMatrix() ;
			if (mode == GL_SELECT)
				glPopName ();
		}
	}

	for (i=0; i<MAXOBJ; i++) {
		if (! Objects[i])
			continue;
		if (i == obj) {
			SetMaterial (1.0, 0.0, 0.0);
			Objects[i]->z += 0.2;
		} else
			SetMaterial (.3, .3, 1.0) ;
		if (Objects[i] == Halo && i != obj)
			Objects[i]->z += 0.2;
		
		if (mode == GL_SELECT)
			glPushName (i);
/*if (mode == GL_SELECT || obj != i)*/
		RenderObject (mode, Objects[i]);
		if (mode == GL_SELECT)
			glPopName ();
		if (i == obj)
			Objects[i]->z -= 0.2;
		if (Objects[i] == Halo && i != obj)
			Objects[i]->z -= 0.2;
	}

	if (obj >= 0 && mode != GL_SELECT)
		RenderShadow (Objects[obj]);
	
	if (Halo && mode != GL_SELECT) {
		tglPushAttrib ();
		tglShadowOffset (0.0, 0.0, 0.05);
		tglShadowColor3d (1.0, 1.0, 0.0);
		tglShadowThickness (0.0, 0.2);
		tglShadowTransparency (0.9, 0.2);
		RenderShadow (Halo);
		tglPopAttrib ();
	}
}

static GLdouble PrevX, PrevY;
static Object* from = 0;
static Object* to = 0;
static int pointno = -1;

void
Module_StartControl(int obj, int x, int y, GLdouble X, GLdouble Y)
{
	Object* o;
	
	PrevX = X;
	PrevY = Y;
	
	if (obj < 0 || ! Objects[obj])
		return;
	o = Objects [obj];
	
	if (o->type == POLY)
		CurPoly = o;
	
	pointno = -1;
	if (o->type == POLY) {
		int i;
		GLdouble dx, dy;
		GLdouble d = 0.25;
		
		for (i = 0; i < o->npoints; i++) {
			dx = o->x + o->X[i] - X;
			dy = o->y + o->Y[i] - Y;
			if (dx > -d && dx < d && dy > -d && dy < d) {
				pointno = i;
				break;
			}
		}
	}
}

void
Module_Control(int obj, int x, int y, GLdouble X, GLdouble Y)
{
	Object* o;
	
	if (obj < 0 || !Objects[obj])
		return;
	o = Objects [obj];
	
	if (o->type == POLY && pointno >= 0) {
		o->X[pointno] += (X - PrevX);
		o->Y[pointno] += (Y - PrevY);
	} else {
		o->x += (X - PrevX);
		o->y += (Y - PrevY);
	}
	
	PrevX = X;
	PrevY = Y;
	glutPostRedisplay ();   
}

void
Module_ControlKey(int obj, int d)
{
	glutPostRedisplay ();   
}

int
Module_keyboardEvents(unsigned char key, int obj, int x, int y, GLdouble X, GLdouble Y)
{
	switch (key) {
	case 'd': NewObj (X, Y, DISK); return 1;
	case 'b': NewObj (X, Y, BOX); return 1;
	case 'f': if (obj >= 0) { from = Objects[obj]; printf ("from is #%d\n", obj); } return 1;
	case 't': if (obj >= 0) { to = Objects[obj]; printf ("to is #%d\n", obj); } return 1;
	case 'e': NewEdge (from, to); return 1;
	case 'w': NewPEdge (from, to, X, Y); return 1;
	case 'P': NewPoly (X, Y); return 1;
	case 'p': if (! CurPoly) NewPoly (X, Y); else AddPoint (X, Y); return 1;
	case '>': if (CurPoly) CurPoly->arrows = TGL_ARROW_END; return 1;
	case '<': if (CurPoly) CurPoly->arrows = TGL_ARROW_START; return 1;
	case '-': if (CurPoly) CurPoly->arrows = TGL_ARROW_NONE; return 1;
	case '=': if (CurPoly) CurPoly->arrows = TGL_ARROW_BOTH; return 1;
	case 'h': if (obj >= 0) { Halo = Objects[obj]; printf ("Halo is #%d\n", obj); } else Halo = 0; return 1;
	}
	
	return 0 ;
}

void
Module_Init( void )
{
  GLfloat ambient[] = {0.2, 0.2, 0.2, 1.0} ;
  GLfloat position[] = {-3.0, 2.0, 5.0, 1.0} ;
  int i;

  tglInit ();
  /*tglBorder (0.0);*/
  
  for (i = 0; i < MAXOBJ; i++) {
	  Objects[i] = 0;
  }

  glClearColor (0.9, 0.9, 0.9, 1.0);
  
/*  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);*/
  glEnable(GL_LIGHTING) ;
  glEnable(GL_LIGHT0) ;
  glLightfv(GL_LIGHT0, GL_AMBIENT, ambient) ;
  glLightfv(GL_LIGHT0, GL_POSITION, position) ;
  
  glMatrixMode(GL_PROJECTION) ;
  glOrtho(-VIEWSIZE, VIEWSIZE, -VIEWSIZE, VIEWSIZE, -VIEWSIZE, VIEWSIZE);
  glMatrixMode(GL_MODELVIEW) ;

  NewObj (0.0, 0.0, DISK);
  
    /* Cree un menu associe au bouton droit */
  glutCreateMenu(menuEvents) ;
  glutAddMenuEntry("Toggle light", 1) ;
  glutAddMenuEntry("Toggle control points", 2) ;
  glutAddMenuEntry("Wireframe-Shaded", 10) ;
  glutAddMenuEntry("-----------------", -1) ;
  glutAddMenuEntry("8 Slices", 100+8) ;
  glutAddMenuEntry("16 Slices", 100+16) ;
  glutAddMenuEntry("30 Slices", 100+30) ;
  glutAddMenuEntry("60 Slices", 100+60) ;
  glutAddMenuEntry("120 Slices", 100+120) ;
  glutAddMenuEntry("-----------------", -1) ;
  glutAddMenuEntry("Zoom 2", 1000+2) ;
  glutAddMenuEntry("Zoom 5", 1000+5) ;
  glutAddMenuEntry("Zoom 10", 1000+10) ;
  glutAddMenuEntry("Zoom 15", 1000+15) ;
  glutAddMenuEntry("Zoom 20", 1000+20) ;
  glutAddMenuEntry("-----------------", -1) ;
  glutAddMenuEntry("Quit", 0) ;

  glutAttachMenu(GLUT_RIGHT_BUTTON) ;

}

