#ifdef WIN32
#include <windows.h>
#endif

#include "glft.h"
#include "freetype.h"


#include <GL/gl.h>
#include <GL/glu.h>

#include <stdio.h>
/* malloc */
#include <stdlib.h>

int glft_ft_error;
int glft_gl_error;


struct glft_font {
  TT_Face* _tt_face;
  TT_CharMap* _tt_charmap;
  TT_Instance*  _tt_instance;
  int ascent, descent, line_gap;
  int units;
};

int
glft_get_ascent(const struct glft_font * f)
{
  return f->ascent;
}

int
glft_get_descent(const struct glft_font * f)
{
  return f->descent;
}


int
glft_get_line_gap(const struct glft_font * f)
{
  return f->line_gap;
}

struct glft_glyph_info {
  int xmin, ymin, xmax, ymax;
  int xadvance;
  int bearing_x, bearing_y;
};


struct glft_contour;

struct glft_glyph {
  struct glft_font* font;
  int num_contours;
  int xadvance;
  int xmin, ymin, xmax, ymax;
  struct glft_contour* contours;
};

struct glft_font*
glft_get_font(struct glft_glyph* g)
{
  return g->font;
}

int
glft_get_xadvance(const struct glft_glyph * g)
{
  return g->xadvance;
}


 

void
glft_get_bounding_box(const struct glft_glyph * g, int* xmin, int* ymin, int* xmax, int* ymax)
{
  *xmin = g->xmin;
  *xmax = g->xmax;
  *ymin = g->ymin;
  *ymax = g->ymax;
}



struct glft_pixmap_glyph {
  struct glft_font* font;
  int xadvance;
  int xmin, ymin, xmax, ymax;
  unsigned char* _pixmap;
};


struct glft_font*
glft_get_pixmap_glyph_font(struct glft_pixmap_glyph* g)
{
  return g->font;
}

int
glft_get_pixmap_glyph_xadvance(const struct glft_pixmap_glyph * g)
{
  return g->xadvance;
}

void
glft_get_pixmap_glyph_bounding_box(const struct glft_pixmap_glyph * g, int* xmin, int* ymin, int* xmax, int* ymax)
{
  *xmin = g->xmin;
  *xmax = g->xmax;
  *ymin = g->ymin;
  *ymax = g->ymax;
}




enum state {
  start=0,
  onstate,
  readingbezier
};

enum transition {
  off=0,
  on
};


struct point2i {
  int x,y;
};


struct line2i {
  struct point2i p1,p2;
};


struct bezier3i {
  struct point2i p1, p2, p3;
};

static
void
print_point2i(FILE* f, struct point2i p)
{
  fprintf(f, "%d %d\n", p.x, p.y);
}

static
void
print_bezier3i(FILE* f, struct bezier3i b)
{
  fprintf(f, "p1: "); print_point2i(stderr, b.p1);
  fprintf(f, "p2: "); print_point2i(stderr, b.p2);
  fprintf(f, "p3: "); print_point2i(stderr, b.p3);
}


static int max_num_point=0;
static int num_point =  0;
static struct point2i* point_list2i = 0;
static char * point_infos = 0;

#define rastnumdiv 3

static
int
subdivide(struct point2i* base)
{

  int a, b;

  base[4].x = base[2].x;
  b = base[1].x;
  a = base[3].x = ( base[2].x + b ) / 2;
  b = base[1].x = ( base[0].x + b ) / 2;
  base[2].x = ( a + b ) / 2;
  
  base[4].y = base[2].y;
  b = base[1].y;
  a = base[3].y = ( base[2].y + b ) / 2;
  b = base[1].y = ( base[0].y + b ) / 2;
  base[2].y = ( a + b ) / 2;

  return 0;
}

static
void
subdivide_bezier(const struct bezier3i *b, struct point2i *ptab, int n)
{
  struct point2i points_stack[(rastnumdiv+1) << 1 + 1];
  int points_stack_top=0;
  int divnum[(rastnumdiv << 1) +1];
  int divnum_top =0;

  int pcount=0;
  int numdiv=1;
  int max;

  if(n==0) {
    ptab[0] = b->p1;
    ptab[1] = b->p3;
    return;
  }

  max = (n < rastnumdiv) ? n : rastnumdiv;
  pcount = (1 << max)-1;

  /* push */
  points_stack[0] = b->p1;
  points_stack[1] = b->p2;
  points_stack[2] = b->p3;
  points_stack_top=0;

  divnum_top=0;
  divnum[0]=0;
  numdiv=0;

  while(points_stack_top+2) {
    while(numdiv < max) {
      /* subdivide_top */
      subdivide(&points_stack[points_stack_top]);
      divnum[divnum_top] = divnum[divnum_top]+1;
      ++divnum_top;
      divnum[divnum_top] = divnum[divnum_top-1];
      points_stack_top+=2;
      ++numdiv;
    }
    
    /* emit */

    ptab[pcount--] = points_stack[points_stack_top+2];
    ptab[pcount--] = points_stack[points_stack_top];

    points_stack_top -= 2;
    numdiv = divnum[divnum_top];
  }
}


struct point {
  double x, y, z;
};

struct glft_contour {
  int num_points;
  struct point *points;
  char * line_infos;
};


static
void
print_point(FILE* f, struct point p)
{
  /*  fprintf(f, "p1: %lf %lf %lf\n", p.x, p.y, p.z);*/
  fprintf(f, "%lf %lf\n", p.x, p.y);
}

static
struct point2i
middle_point2i(struct point2i p1, struct point2i p2)
{
  struct point2i p;
  p.x = (p1.x+p2.x) >> 1;
  p.y = (p1.y+p2.y) >> 1;
  return p;
}

static
struct point
point2i_to_point(struct point2i p2i)
{
  struct point p;
  p.x = p2i.x;
  p.y = p2i.y;
  p.z = 0;
  return p;
}

static
void
scale(struct point* p, double v)
{
  p->x*=v;
  p->y*=v;
  p->z*=v;
}


static GLUtesselator* thetess;
static TT_Engine _tt_engine;
static int inited=0;

/* CALLBACK is for windoze users... not SGI ones!
 Thanks to G. Lanois (gerard@msi.com)
 */
#if !defined(CALLBACK) && !defined(__WIN32__) && !defined(__WINDOWS__)
#define CALLBACK
#endif

#define CALLBACKARG void


static
void CALLBACK beginCallback(GLenum which)
{
  glBegin(which);
}

static
void CALLBACK endCallback(void)
{
  glEnd();
}

static
void CALLBACK errorCallback(GLenum errorCode)
{
  const GLubyte *estring;

  estring = gluErrorString(errorCode);
  fprintf (stderr, "Tessellation Error: %s\n", estring);
  exit (0);
}

static
void CALLBACK combineCallback(GLdouble coords[3],
                              GLdouble* vertex_data[4],
                              GLfloat weight[4], GLdouble **dataOut )
{
  /* I don't know how to use the combine Callbak */
  /* I just coded something that seems to work... */
#if 1
  int i;
  *dataOut = (GLdouble*) malloc(4*sizeof(double));
  for(i=0;i<4;++i) {
    (*dataOut)[i] = coords[i];

#if 0
    int j;
    (*dataOut)[i] = 0;
    for(j=0;j<4;++j)
      *dataOut[i] += vertex_data[i][j];
    (*dataOut)[i] /= 4.0;
#endif

  }
#else

  *dataOut = coords;
#endif
   
}

#if defined(WIN32) && !defined(__CYGWIN32__)
typedef void (CALLBACK *glu_callback)(CALLBACKARG);
#else
typedef void CALLBACK (*glu_callback)(CALLBACKARG);
#endif


int
glft_init(void)
{
  int error;
  glft_ft_error = 0;
  glft_gl_error = 0;


  point_list2i = 0;
  max_num_point = 3000;

  point_list2i = (struct point2i*) malloc(sizeof(struct point2i) * max_num_point);
  point_infos = (char*) malloc(sizeof(char) * max_num_point);

  thetess = gluNewTess();
  if(!thetess) {
    glft_gl_error = glGetError();
    return glft_gl_error;
  }


  gluTessCallback(thetess, GLU_TESS_VERTEX,
                  (glu_callback) &glVertex3dv);
  gluTessCallback(thetess, GLU_TESS_BEGIN,
                  (glu_callback) &beginCallback);
  gluTessCallback(thetess, GLU_TESS_END,
                  (glu_callback) &endCallback);
  gluTessCallback(thetess, GLU_TESS_ERROR,
                  (glu_callback) &errorCallback);
  
  gluTessCallback(thetess, GLU_TESS_COMBINE,
                  (glu_callback) &combineCallback);
                   

  gluTessProperty(thetess, GLU_TESS_WINDING_RULE,  GLU_TESS_WINDING_NONZERO);
  gluTessNormal(thetess, 0, 0, 1);

  glft_gl_error = glGetError();
  if(glft_gl_error)
    return glft_gl_error;


  error = TT_Init_FreeType( &_tt_engine );
  if (error) {
    fprintf(stderr, "could not create engine instance\n");
    return error;
  }
  inited=1;
  return 0;
}

int
glft_done(void)
{
  gluDeleteTess(thetess);
  glft_gl_error = glGetError();
  if(glft_gl_error)
    return glft_gl_error;
  if(point_list2i)
    free(point_list2i);
  if(point_infos)
    free(point_infos);

  return TT_Done_FreeType(_tt_engine);
}


struct glft_font*
glft_new_font(const char* file)
{
  int error=0;
  int i,n;
  TT_Face_Properties  properties;
  struct glft_font* thefont = (struct glft_font*) malloc(sizeof(struct glft_font));

  if(!inited) {
    glft_init();
    inited=1;
  }

  thefont->_tt_face = (TT_Face*) malloc(sizeof(TT_Face));
  error = TT_Open_Face( _tt_engine, file, thefont->_tt_face );
  if (error) {
    fprintf(stderr, "could not open font file %s, error %d\n", file, error);
    return 0;
  }


  
  n = TT_Get_CharMap_Count(*thefont->_tt_face);
  
  
 
  thefont->_tt_charmap = (TT_CharMap*) malloc(sizeof(TT_CharMap));

  for( i= 0; i < n; ++i ) {
    short unsigned platform= 0;
    short unsigned encoding= 0;
    TT_Get_CharMap_ID( *thefont->_tt_face, i, &platform, &encoding );
    if ( (platform == 3 && encoding == 1 )  || (platform == 0 && encoding == 0 ) ) {
      error = TT_Get_CharMap( *thefont->_tt_face, i, thefont->_tt_charmap );
      if(error) {
        fprintf(stderr, "char_map error\n");
        return 0;
      }
      break;
    }
  }


  /*
    if(properties.max_Points > max_num_point) {
    max_num_point = properties.max_Points;
    fprintf (stderr, "%d %d\n", max_num_point, 1 << rastnumdiv);
    if(point_list2i) {
    free(point_list2i);
    }
    point_list2i = (struct point2i*) malloc(sizeof(struct point2i) * max_num_point * (1 << rastnumdiv+1));
    }
  */

  

  thefont->_tt_instance = (TT_Instance*) malloc(sizeof(TT_Instance)); 
  error = TT_New_Instance( *thefont->_tt_face, thefont->_tt_instance );
  if (error) {
    fprintf(stderr, "could not open instance\n");
    return 0;
  }
  error = TT_Set_Instance_CharSize( *thefont->_tt_instance, 12*64 );
  
  error = TT_Set_Instance_Resolutions(*thefont->_tt_instance, 72, 72);
  
  
  TT_Get_Face_Properties( *thefont->_tt_face, &properties );

  
  thefont->ascent = properties.horizontal->Ascender;
  thefont->descent = properties.horizontal->Descender;
  thefont->line_gap = properties.horizontal->Line_Gap;
  thefont->units = properties.header->Units_Per_EM;
  
  return thefont;
}


void
glft_delete_font(struct glft_font* thefont)
{
        
  TT_Close_Face(*thefont->_tt_face);
  free(thefont->_tt_charmap);
  free(thefont->_tt_instance);
  free(thefont->_tt_face);
  free(thefont);
}


static
int
new_read_points(TT_Vector *points, TT_Byte* flags, int first, int last)
{
  enum state current = start;
  enum transition trans=off;
  int i, j, end;
  struct line2i theline;
  struct bezier3i thebezier;
  struct point2i p;

  num_point=0;

  end=0;
  for(i=first; !end && i<=last+1; ++i) {
    int t;
    if(i==last+1) {
      i=first;
      end=1;
    }
    t = flags[i] & 0x01;
    if(t==0)
      trans = off;
    else if(t==1)
      trans = on;
    else {
      fprintf(stderr, "transition error\n");
      return 1;
    }
    switch(current) {
    case start:
      switch (trans) {
      case on:
        p.x = points[i].x;
        p.y = points[i].y;

        /* points */
        if(num_point==max_num_point) {
          fprintf(stderr, "read_point too many points %d\n", i);
          return 1;
        }
        point_list2i[num_point]=p;
        ++num_point;

        current = onstate;
        break;
      case off:
        fprintf(stderr, "read_point error %d\n", i);
        return 1;
      }
      break;
    case onstate:
      switch (trans) {
      case on:
        /*      fprintf(stderr, "line\n");*/
        theline.p1 = p;
        theline.p2.x = points[i].x;
        theline.p2.y = points[i].y;

        p = theline.p2;

        /* points */
        if(num_point==max_num_point) {
          fprintf(stderr, "read_point too many points %d\n", i);
          return 1;
        }
        point_list2i[num_point]=p;
        point_infos[num_point]=1;
        ++num_point;

        break;

      case off:
        thebezier.p1 = p;
        thebezier.p2.x = points[i].x;
        thebezier.p2.y = points[i].y;
        
        p = thebezier.p2;
        current = readingbezier;
        break;
      }
      break;
    case readingbezier:
      switch (trans) {
      case on:
        /*      fprintf(stderr, "simple bezier\n");*/
        thebezier.p3.x = points[i].x;
        thebezier.p3.y = points[i].y;
        p = thebezier.p3;

        /* points */
        if(num_point+(1<<rastnumdiv) >=max_num_point) {
          fprintf(stderr, "read_point too many points %d\n", i);
          return 1;
        }

        subdivide_bezier(&thebezier, &point_list2i[num_point], rastnumdiv);
        for(j=num_point; j<num_point+(1<< rastnumdiv); ++j)
          point_infos[j]=0;
        num_point+= 1 << rastnumdiv;

        current = onstate;
        break;
      case off:
        /*      fprintf(stderr, "multi bezier\n");*/
        p.x = points[i].x;
        p.y = points[i].y;
        thebezier.p3 = middle_point2i(thebezier.p2,p);

        /* points */
        if(num_point+(1<<rastnumdiv) >=max_num_point) {
          fprintf(stderr, "read_point too many points %d\n", i);
          return 1;
        }

        subdivide_bezier(&thebezier, &point_list2i[num_point], rastnumdiv);
        for(j=num_point; j<num_point+(1<< rastnumdiv); ++j)
          point_infos[j]=0;
        num_point+= 1 << rastnumdiv;

        thebezier.p1= thebezier.p3;
        thebezier.p2 = p;

        
        break;
      }
      break;
    default:
      break;
    }
  }
  return 0;
}

int
glft_get_glyph_info(struct glft_font* thefont, char thechar, int size, struct glft_glyph_info* info)
{
  int error;
  TT_Glyph _tt_glyph;
  TT_Glyph_Metrics metrics;
  TT_Outline _tt_outline;
  int i;

  error = TT_Set_Instance_CharSize( *thefont->_tt_instance, size*64 );
  if (error) {
    fprintf(stderr, "could not set instance char size\n");
    return -1;
  }

  error = TT_New_Glyph(  *thefont->_tt_face, &_tt_glyph );
  if (error) {
    fprintf(stderr, "could not create glyph\n");
    return -1;
  }

  error = TT_Load_Glyph( *thefont->_tt_instance, _tt_glyph, TT_Char_Index( *thefont->_tt_charmap, thechar), TTLOAD_DEFAULT);
  if(error) {
    fprintf(stderr, "load glyph error\n");
    return -1;
  }


  error = TT_Get_Glyph_Metrics( _tt_glyph, &metrics );
  if (error) {
    fprintf(stderr, "could not get metrics\n");
    return -1;
  }

  info->xmax = metrics.bbox.xMax >> 6;
  info->xmin = metrics.bbox.xMin >> 6;
  info->ymax = metrics.bbox.yMax >> 6;
  info->ymin = metrics.bbox.yMin >> 6;

  info->xadvance = metrics.advance >> 6;
  info->bearing_x = metrics.bearingX >> 6;
  info->bearing_y = metrics.bearingY >> 6;
  return 0;
}

int glft_get_glyph_metrics(struct glft_font* thefont, char thechar, int size, int *xadvance)
{
  struct glft_glyph_info info;
  int err;
  
  err = glft_get_glyph_info(thefont, thechar, size, &info);
  if (err) {
    return err;
  }
  
  *xadvance = info.xadvance;
  return 0;
}

struct glft_glyph*
glft_new_glyph(struct glft_font* thefont, char thechar, int size)
{
  int error;
  TT_Glyph _tt_glyph;
  TT_Glyph_Metrics metrics;
  TT_Outline _tt_outline;
  struct glft_glyph theglyph;
  struct glft_glyph* rglyph;
  int i;

  error = TT_Set_Instance_CharSize( *thefont->_tt_instance, size*64 );
  if (error) {
    fprintf(stderr, "could not set instance char size\n");
    return 0;
  }

  error = TT_New_Glyph(  *thefont->_tt_face, &_tt_glyph );
  if (error) {
    fprintf(stderr, "could not create glyph\n");
    return 0;
  }

  error = TT_Load_Glyph( *thefont->_tt_instance, _tt_glyph, TT_Char_Index( *thefont->_tt_charmap, thechar), TTLOAD_DEFAULT);
  if(error) {
    fprintf(stderr, "load glyph error\n");
    return 0;
  }


  error = TT_Get_Glyph_Metrics( _tt_glyph, &metrics );
  if (error) {
    fprintf(stderr, "could not get metrics\n");
    return 0;
  }

  theglyph.xmax = metrics.bbox.xMax >> 6;
  theglyph.xmin = metrics.bbox.xMin >> 6;
  theglyph.ymax = metrics.bbox.yMax >> 6;
  theglyph.ymin = metrics.bbox.yMin >> 6;

  theglyph.xadvance = metrics.advance >> 6;
    
  error = TT_Get_Glyph_Outline(_tt_glyph, &_tt_outline);
  if (error) {
    fprintf(stderr, "could not get outline\n");
    return 0;
  }

  rglyph = (struct glft_glyph*) malloc(sizeof(struct glft_glyph));

  {
    int first_point=0;
    int toto=0;
    theglyph.num_contours = _tt_outline.n_contours;
    theglyph.contours = (struct glft_contour*) malloc( sizeof(struct glft_contour) * theglyph.num_contours);

    for(i=0; i<_tt_outline.n_contours; ++i) {
      int j;
      /*    num_point = 0;*/
      new_read_points(_tt_outline.points,_tt_outline.flags, first_point,_tt_outline.contours[i] );
      toto+=num_point;

      theglyph.contours[i].num_points = num_point;
      theglyph.contours[i].points = (struct point*) malloc (sizeof (struct point) * theglyph.contours[i].num_points);
      theglyph.contours[i].line_infos = (char*) malloc (sizeof (char) * theglyph.contours[i].num_points);
      for(j=0; j<num_point; ++j) {
        /*      theglyph.contours[i].points[j] = point_list[j];*/
        theglyph.contours[i].points[j] = point2i_to_point(point_list2i[j]);
        scale(&theglyph.contours[i].points[j],1/64.);
        theglyph.contours[i].line_infos[j] = point_infos[j];
      }
    
      first_point = _tt_outline.contours[i]+1;
    }
  }

  *rglyph = theglyph;
  return rglyph;
}

void
glft_delete_glyph(struct glft_glyph* theglyph)
{
  int i;
  for(i=0; i<theglyph->num_contours; ++i) {
    free(theglyph->contours[i].points);
  }
  free(theglyph->contours);
  free(theglyph);
}



void
glft_glyph_render_plain(const struct glft_glyph* theglyph)
{
  int i;
  gluTessProperty(thetess,  GLU_TESS_BOUNDARY_ONLY, GL_FALSE);
  gluTessBeginPolygon(thetess,0);
  for(i=0; i< theglyph->num_contours; ++i) {
    int j;
    gluTessBeginContour(thetess);
    for(j=0; j<theglyph->contours[i].num_points; ++j) {
      gluTessVertex(thetess, (GLdouble*)&theglyph->contours[i].points[j], &theglyph->contours[i].points[j]);
    }
    gluTessEndContour(thetess);
  }
  gluTessEndPolygon(thetess);

#if 0
  /* anti-aliasing */

  glEnable(GL_LINE_SMOOTH);
  for(i=0; i< theglyph->num_contours; ++i) {
    struct point p1,p2;
    int j;
    if(theglyph->contours[i].num_points) {
      p1 = theglyph->contours[i].points[0];
      glBegin(GL_LINE_STRIP);
      glVertex2d(p1.x,p1.y);
      for(j=1; j<theglyph->contours[i].num_points; ++j) {
        double x,y;
        x = p1.x-theglyph->contours[i].points[j].x;
        y = p1.y-theglyph->contours[i].points[j].y;
        
        if(theglyph->contours[i].line_infos[j] == 1 && ( (x*x<1) || (y*y<1))) {
          while(j < theglyph->contours[i].num_points && theglyph->contours[i].line_infos[j]) {
            ++j;
            x = p1.x-theglyph->contours[i].points[j].x;
            y = p1.y-theglyph->contours[i].points[j].y;
            if( !((x*x<1) && (y*y<1)) ) 
              break;
          }
          if (j == theglyph->contours[i].num_points)
            break;
          glEnd();
          glBegin(GL_LINE_STRIP);
          --j;
        }
#if 1
        else
          if((x*x<1) && (y*y<1)) {
            while(j < theglyph->contours[i].num_points ) {
              ++j;
              x = p1.x-theglyph->contours[i].points[j].x;
              y = p1.y-theglyph->contours[i].points[j].y;
              if( !((x*x<1) && (y*y<1)) ) 
                break;
            }
            if (j == theglyph->contours[i].num_points)
              break;
            glEnd();
            glBegin(GL_LINE_STRIP);
            --j;
          }
#endif      

        p1 = theglyph->contours[i].points[j];
      
        glVertex2d(p1.x,p1.y);
      }
      glEnd();

    }
  }
  glDisable(GL_LINE_SMOOTH);
#endif
}

void
glft_glyph_render_outline(const struct glft_glyph* theglyph)
{
  int i;
  for(i=0; i< theglyph->num_contours; ++i) {
    int j;
    glBegin(GL_LINE_STRIP);
    for(j=0; j<theglyph->contours[i].num_points; ++j) {
      glVertex2d(theglyph->contours[i].points[j].x, theglyph->contours[i].points[j].y);
    }
    glEnd();
  }
}

struct glft_pixmap_glyph*
glft_new_pixmap_glyph(struct glft_font* thefont, char thechar, int size)
{
  int error;
  TT_Glyph _tt_glyph;
  TT_Glyph_Metrics metrics;
  TT_Raster_Map _tt_map;
  struct glft_pixmap_glyph theglyph;
  struct glft_pixmap_glyph* rglyph;
  int i,j;
  unsigned char palette[5][2];
  int bytes_per_texel;
  int width, height, cols;
  char *bitmap_it, *pixmap_it;
  char *pixmap;


  error = TT_Set_Instance_CharSize( *thefont->_tt_instance, size*64 );
  if (error) {
    fprintf(stderr, "could not set instance char size\n");
    return 0;
  }

  error = TT_New_Glyph(  *thefont->_tt_face, &_tt_glyph );
  if (error) {
    fprintf(stderr, "could not create glyph\n");
    return 0;
  }

  error = TT_Load_Glyph( *thefont->_tt_instance, _tt_glyph, TT_Char_Index( *thefont->_tt_charmap, thechar), TTLOAD_DEFAULT);
  if(error) {
    fprintf(stderr, "load glyph error\n");
    return 0;
  }


  error = TT_Get_Glyph_Metrics( _tt_glyph, &metrics );
  if (error) {
    fprintf(stderr, "could not get metrics\n");
    return 0;
  }

#define  FLOOR(x)    ((x) & -64)
#define  CEILING(x)  (((x)+63) & -64)

  theglyph.xmax = CEILING(metrics.bbox.xMax) >> 6;
  theglyph.xmin = FLOOR(metrics.bbox.xMin) >> 6;
  theglyph.ymax = CEILING(metrics.bbox.yMax) >> 6;
  theglyph.ymin = FLOOR(metrics.bbox.yMin) >> 6;

#undef CEILING
#undef FLOOR

  theglyph.xadvance = metrics.advance >> 6;

#ifdef __luminance_alpha__
  bytes_per_texel=2;
#else
  bytes_per_texel=1;
#endif
  
  for(j= 0; j < 5; ++j ) {
    palette[j][0]= 255; /* 0 dark, 255 if in texture because intensity modulates the frag */
    palette[j][1]= (unsigned char)(255 * j / 4.0);
  }
  /* palette[0][0] = 0; */ /* 255 */

  width = theglyph.xmax - theglyph.xmin;
  height = theglyph.ymax - theglyph.ymin;
  cols= (width+3) & -4;


  pixmap = malloc(cols * height);
  memset( (void*) pixmap, 0, cols * height );
  
  _tt_map.width = width;
  _tt_map.cols  = cols;
  _tt_map.rows  = height;
  _tt_map.flow  = TT_Flow_Up;
  _tt_map.size  = cols * height;
  _tt_map.bitmap= (void*) pixmap;
  
  error = TT_Get_Glyph_Pixmap( _tt_glyph, &_tt_map, -metrics.bbox.xMin, -metrics.bbox.yMin );
  
  if (error) {
    fprintf(stderr,"could not get pixmap\n");
    return 0;
  }

  theglyph._pixmap = malloc(width * height * bytes_per_texel);
  memset( (void*) theglyph._pixmap, 0, width * height * bytes_per_texel);


  bitmap_it = pixmap;
  pixmap_it = theglyph._pixmap;
  for( i=0; i<height; ++i ) {
    for (j=0; j<width; ++j) {
    
    
    
#ifdef __luminance_alpha__
      *pixmap_it++=palette[*bitmap_it][0];
#endif
      *pixmap_it++=palette[*bitmap_it++][1];
    }
    bitmap_it+=(cols-width);
    /*    p+= bytes_per_texel;*/
  }


  free(pixmap);

  rglyph = (struct glft_pixmap_glyph*) malloc(sizeof(struct glft_pixmap_glyph));
  *rglyph = theglyph;

  return rglyph;
}

void
glft_delete_pixmap_glyph(struct glft_pixmap_glyph* theglyph)
{
  free(theglyph->_pixmap);
  free(theglyph);
}

char*
glft_get_pixmap_glyph_pixmap(const struct glft_pixmap_glyph * g) {
  return g->_pixmap;
}


void
glft_pixmap_glyph_render(struct glft_pixmap_glyph* theglyph)
{ 
  glRasterPos2i(theglyph->xmin, theglyph->ymin);
  glDrawPixels(theglyph->xmax-theglyph->xmin, theglyph->ymax-theglyph->ymin, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, theglyph->_pixmap);
}


struct texture_glyph_info {
  int x1, y1, x2, y2, xadvance, bearing_x, bearing_y;
};

struct glft_pixmap_texture {
  int width, height;
  char* pixmap;
  struct texture_glyph_info* glyph_infos[256];
};


int
glft_get_pixmap_texture_width(struct glft_pixmap_texture* tex)
{
  return tex->width;
}

int
glft_get_pixmap_texture_height(struct glft_pixmap_texture* tex)
{
  return tex->height;
}

char*
glft_get_pixmap_texture_pixmap(struct glft_pixmap_texture* tex)
{
  return tex->pixmap;
}


struct glft_pixmap_texture*
glft_new_pixmap_texture(struct glft_font* thefont, const char* charset, int thesize)
{
  int i=0, err = 0;
  char done[256];
  struct texture_glyph_info infos[256];
  const char * char_it;

  int width=0, height=0;
  int wgap=2, hgap=2;
  int max_size;
  int max_size_reached=0;
  int max_height=0;
  int bytes_per_texel=2;

  struct glft_pixmap_texture* thetexture;

  glGetIntegerv(GL_MAX_TEXTURE_SIZE, &max_size);

  for(i=0; i<256; ++i) {
    done[i]=0;
  }

  /* first find the texture size */
  char_it = charset;
  while(*char_it) {
    if(!done[*char_it]) {
      int w,h;
      struct glft_glyph_info theinfo;
      err = glft_get_glyph_info(thefont, *char_it, thesize, &theinfo);
      if(err) {
        fprintf(stderr, "%s l:%d - can't get glyph info", __FILE__, __LINE__);
        return 0;
      }
      w = theinfo.xmax - theinfo.xmin;
      h = theinfo.ymax - theinfo.ymin;

      infos[*char_it].x1 = width;
      infos[*char_it].y1 = height;

      infos[*char_it].xadvance = theinfo.xadvance;
      infos[*char_it].bearing_x = theinfo.bearing_x;
      infos[*char_it].bearing_y = theinfo.bearing_y;
      
      if(width + w + wgap > max_size) {
        width = 0;
        max_size_reached=1;
        height += max_height + hgap;
        if(height>max_size || w>max_size) {
          fprintf(stderr, "texture too large\n");
          return 0;
        }
        infos[*char_it].x1 = width;
        infos[*char_it].y1 = height;
        max_height = h;
      }
      else {
        if(h>max_height)
          max_height=h;
      }
      width += w + wgap;
      infos[*char_it].x2=infos[*char_it].x1+w;
      infos[*char_it].y2=infos[*char_it].y1+h;      


      done[*char_it] = 1;
    }
    ++char_it;
  }

  if(max_size_reached)
    width = max_size;
  height+= max_height;
  if(height>max_size) {
    fprintf(stderr, "%s l:%d - texture too large (%dx%d) max:%d", __FILE__, __LINE__, width,height,max_size);
    return 0;
  }

  { /* dimensions as a power of 2 */
    int h=1;
    int w=1;
    while(h<height)
      h<<=1;
    while(w<width)
      w<<=1;
    if(w>max_size || h>max_size) {
      fprintf(stderr, "%s l:%d - texture too large (%dx%d) max:%d", __FILE__, __LINE__, w,h,max_size);
      return 0;
    }
    width=w;
    height=h;
    /*    fprintf(stderr, "%d %d\n", width, height);*/
  }


  /* then render... */
#ifdef __luminance_alpha__
  bytes_per_texel=2;
#else
  bytes_per_texel=1;
#endif

  thetexture = (struct glft_pixmap_texture*) malloc(sizeof(struct glft_pixmap_texture));
  thetexture->pixmap = malloc (width * height * bytes_per_texel);
  memset( (void*) thetexture->pixmap, 0, width * height * bytes_per_texel );

  thetexture->width=width;
  thetexture->height=height;

  for(i=0; i<256; ++i) {
    if(done[i]) {
      int j,k;
      char *from, *to;
      struct glft_pixmap_glyph* theglyph;

      thetexture->glyph_infos[i] = (struct texture_glyph_info*) malloc(sizeof(struct texture_glyph_info));
      *thetexture->glyph_infos[i] = infos[i];

      /*fprintf(stderr, "%d %d %d %d\n", infos[i].x1, infos[i].y1, infos[i].x2, infos[i].y2 );*/

      theglyph = glft_new_pixmap_glyph(thefont, i, thesize);

      from = theglyph->_pixmap;
      to = thetexture->pixmap + (infos[i].y1 * width + infos[i].x1) * bytes_per_texel;
      for(j=0; j<(infos[i].y2-infos[i].y1); ++j) {
        for(k=0; k<infos[i].x2-infos[i].x1; ++k) {
#ifdef __luminance_alpha__
          *to++ = *from++;
#endif
          *to++ = *from++;
        }
        to+= (width - (infos[i].x2-infos[i].x1))* bytes_per_texel;
      }

      glft_delete_pixmap_glyph(theglyph);
    }
    else
      thetexture->glyph_infos[i]=0;
  }

  return thetexture;
}

void
glft_delete_pixmap_texture(struct glft_pixmap_texture* tex)
{
  int i;

  for(i=0; i<256; ++i)
    if(tex->glyph_infos[i])
      free(tex->glyph_infos[i]);
  free(tex->pixmap);
  free(tex);
}

void
glft_pixmap_texture_render_texture(struct glft_pixmap_texture* tex)
{
#ifdef __luminance_alpha__
  glDrawPixels(tex->width, tex->height, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, tex->pixmap);
#else
  glDrawPixels(tex->width, tex->height, GL_ALPHA, GL_UNSIGNED_BYTE, tex->pixmap);
#endif
}

void
glft_pixmap_texture_render(struct glft_pixmap_texture* thetex, const char* thestring)
{
  struct texture_glyph_info info;
  float xadvance = 0;

  glBegin(GL_QUADS);
  while(*thestring) {
    float x;
    info = * (thetex->glyph_infos[*thestring]);
    x = info.bearing_x+xadvance;

    glTexCoord2f(info.x1,info.y1);      glVertex2f(x,info.bearing_y-(info.y2-info.y1));
    glTexCoord2f(info.x1,info.y2);      glVertex2f(x,info.bearing_y);
    glTexCoord2f(info.x2,info.y2);      glVertex2f(x+(info.x2-info.x1),info.bearing_y);
    glTexCoord2f(info.x2,info.y1);      glVertex2f(x+(info.x2-info.x1),info.bearing_y-(info.y2-info.y1));

    xadvance += info.xadvance;

    ++thestring;
  }
  glEnd();

}


#if 0
info.advance= metrics.advance/64;
info.bearing_y = metrics.bearingY/64;
info.bearing_x = metrics.bearingX/64; 
        
#define  FLOOR(x)    ((x) & -64)
#define  CEILING(x)  (((x)+63) & -64)
bbox.xMin= FLOOR(bbox.xMin);
bbox.yMin= FLOOR(bbox.yMin);
bbox.xMax= CEILING(bbox.xMax);
bbox.yMax= CEILING(bbox.yMax);
#undef CEILING
#undef FLOOR
info.width = (bbox.xMax - bbox.xMin)/64;
info.height= (bbox.yMax - bbox.yMin)/64;
info.cols= (info.width+3) & -4;



#define __luminance_alpha__


_map = new unsigned char[(int)(_texture_width * _texture_height * bytes_per_texel)];
memset(_map, 0, _texture_width * _texture_height * bytes_per_texel);

unsigned char palette[5][2];
for( int j= 0; j < 5; ++j ) {
  palette[j][0]= 255; // 0
  palette[j][1]= (unsigned char)(255 * j / 4.0);

}
//      palette[0][0] = 0; // 255

unsigned char* bitmap=0;
int bitmap_size=0;
glyph = glyphes;
unsigned char* p = _map;
int num_glyph=0;
int line_glyph = 0;
maxh=0;
int totalh=0;

        
while(*glyph) {
  glyph_info info = *_infos[*glyph];
  int size2 = info.cols * info.height;

  if(size2>bitmap_size) {
    delete [] bitmap;
    bitmap= new unsigned char [ size2 ];
    bitmap_size = size2;

  }
  memset( (void*) bitmap, 0, bitmap_size );     

        
  TT_Raster_Map _tt_map;
  _tt_map.width = info.width;
  _tt_map.cols  = info.cols;
  _tt_map.rows  = info.height;
  _tt_map.flow  = TT_Flow_Up;
  _tt_map.size  = size2;
  _tt_map.bitmap= (void*) bitmap;
        
  error = TT_Get_Glyph_Pixmap( *_tt_glyphes[*glyph], &_tt_map, -_bboxes[*glyph].xMin, -_bboxes[*glyph].yMin );
  if (error) {
    cerr << error << " could not get pixmap" << endl;
  }

  unsigned char* b = bitmap;
  unsigned char* p1 = p;
  for( int i=0; i< info.height; ++i ) {
    for (int j=0; j<info.width; ++j) {
#ifdef __luminance_alpha__
      *p++=palette[*b][0];
#endif
      *p++=palette[*b++][1];
    }

    b+=(info.cols-info.width);
    p+= (int)((_texture_width-info.width) * bytes_per_texel);
  }

  p=p1+(int)((info.width+xgap) * bytes_per_texel);       // +2: space between glyphes to avoid artifacts
  _infos[*glyph]->x1 = num_glyph;
  _infos[*glyph]->y1 = totalh ;
  _infos[*glyph]->x2 = _infos[*glyph]->x1 + (info.width);
  _infos[*glyph]->y2 = (totalh + info.height);

  num_glyph += info.width+xgap;
  if(info.height > maxh)
    maxh = info.height;

  ++glyph;              
  if( *glyph && (num_glyph + _infos[*glyph]->width+xgap)>_texture_width ) {
    num_glyph = 0;
    maxh+=ygap; // +4: space between glyphes to avoid artifacts
    totalh+=maxh;
    ++line_glyph;
    p = &_map[(int)(_texture_width*totalh * bytes_per_texel)];
    maxh=0;
  }

}
#endif
