#ifndef _hapi_h
#define _hapi_h

#include <windows.h>

// TODO use enum instead of defines

/// The following are flags you can give to hInitialise
#define NONE 0
/// Send event on button press/release
#define BUTTON 0x1
/// Send event on mouse movement
#define MOVEMENT 0x2
/// Draw cursors
#define CURSOR 0x4
/// Keep cursor within a rectangle
#define CLIP 0x8
/// Should the API perform acceleration with the settings
// from Windows?
#define ACCELERATE 0x10
/// Should we send events for a suspended mouse
#define SUSPEND 0x20

/** All functions defined herein are thread-safe except
 * for hInitialise; See commest for hInitialise for more
 * information.
 */

/** This is a point used for getting/setting when more
 * accuracy is required in return parameter.
 */
typedef struct hPOINT {
	int x, y;
} hPOINT;

/** Initialise the API. This must be called or all the
 * other functions will return abnormally. After this
 * returns from the first call, it is thread-safe.
 * @param count - Number of mice to try to acquire
 * @param window - Window to send events to. Only required
 * if either of BUTTON or MOVEMENT are specified
 * @param screen - The display context to draw cursors to.
 * Only required if either of CURSOR or CLIP are specified
 * @param flags - Specifies flags as described above
 * @return the number of mice actually acquired
 */
int __cdecl hInitialise(int count,
		HWND window,
		HDC screen,
		unsigned int flags);

/** Shutdown the API. Takes care of deallocating allocated
 * ressources, including killing listener thread. It also
 * UnGets all acquired mice.
 */
void __cdecl hCleanup(void);

/** Suspend the specified mouse
 * @param number - The mouse to suspend
 */
void __cdecl hSuspendMouse(int number);

/** UnSuspend the specified mouse
 * @param number - The mouse to unsuspend
 */
void __cdecl hUnSuspendMouse(int number);

/** Get the relative movement of the specified mouse since
 * the last call. This is not affected by the CLIP flag.
 * @param number - The mouse for which to get the relative
 * position
 * @param p - a pointer to a hPOINT to store the position
 * in
 */
void __cdecl hGetRelativePosition(int number, hPOINT* p);

/** Get the absolute position of the specified mouse. This
 * is affected by the CLIP flag (ie. the position is
 * always within the DC given to hInitialise).
 * @param number - The mouse for which to get the position
 * @param p - a pointer to a hPOINT to store the position
 * in
 */
void __cdecl hGetAbsolutePosition(int number, hPOINT* p);

/** Set the relative position of the specified mouse. This
 * is clipped to the DC given to hInitialise is the CLIP
 * flag was specified. If the CURSOR flag was specified,
 * the cursors are redrawn.
 * @param number - The mouse to set the position for
 * @param p - Where to move the mouse
 */
void __cdecl hSetAbsolutePosition(int number, hPOINT p);

/** Set a cursor for a given mouse. If the CURSOR flag is
 * not given this only stores the cursor, but does not
 * display it.
 * @param number - The mouse to set the cursor for
 * @param p - The hotspot of the cursor
 * @param cursor -a bitmap describing the cursor. Set to
 * null to hide the cursor.  Areas with color (255, 0, 205)
 * are transparent, areas with color (255, 0, 204) are replaced
 * by the colour specified
 * @param r, g, b - RGB values of color to place instead
 * of pixels with index 1
 */
void __cdecl hSetCursor(int number,
		hPOINT p,
		HBITMAP cursor,
		BYTE r, BYTE g, BYTE b);

/** Lock the canvas on which cursors are drawn. No cursors
 * will be updated when this is called until
 * hUpdateCursors is called manually from within the same
 * thread.
 */
void __cdecl hLockCanvas();

/** Redisplay all cursors (only if the CURSOR flag is
 * given)
 * @param usecache - Swap the cache of the background in
 * before redrawing cursors. Normally you want this, but
 * if you know that you have redrawn the DC on which the
 * cursors are drawn, this can corrupt the image.
 */
void __cdecl hUpdateCursors(int useCache);

int __cdecl hGetLastSuspended(void);
int __cdecl hGetLastUnSuspended(void);
int __cdecl hGetLastMoved(void);

#endif // ifndef _hapi_h
